<?php

declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;

class VideoTempsTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('video_temp');
		$this->setPrimaryKey('id');

		$this->belongsTo('Users', [
			'foreignKey' => 'user_id',
			'className' => 'App\Model\Table\UsersTable',
		]);

		$this->belongsTo('Videos', [
			'foreignKey' => 'video_id',
			'className' => 'App\Model\Table\VideosTable',
		]);
	}

	public function getDetails(int $id): array
	{
		return $this->find()
			->where(['VideoTemp.id' => $id])
			->enableHydration(false)
			->first() ?? [];
	}

	public function getDetailsAgainstIDAndUserID(int $user_id, int $id): array
	{
		return $this->find()
			->where([
				'VideoTemp.id' => $id,
				'VideoTemp.user_id' => $user_id,
			])
			->enableHydration(false)
			->first() ?? [];
	}

	public function getOldRecords(): array
	{
		return $this->find()
			->where([
				'VideoTemp.created <' => date('Y-m-d H:i:s', strtotime('-10 minutes'))
			])
			->enableHydration(false)
			->all()
			->toList();
	}

	public function countWatchVideos(array $video_ids, string $start_datetime, string $end_datetime): int
	{
		return (int)($this->getConnection()
			->newQuery()
			->select(['count' => 'COUNT(*)'])
			->from('video_watch')
			->where([
				'video_id IN' => $video_ids,
				'created >=' => $start_datetime,
				'created <=' => $end_datetime,
			])
			->execute()
			->fetch('assoc')['count'] ?? 0);
	}

	public function countWatchVideosByDate(array $video_ids, string $start_datetime, string $end_datetime): array
	{
		return $this->getConnection()
			->newQuery()
			->select([
				'date' => 'DATE(created)',
				'count' => 'COUNT(*)'
			])
			->from('video_watch')
			->where([
				'video_id IN' => $video_ids,
				'created >=' => $start_datetime,
				'created <=' => $end_datetime,
			])
			->group('DATE(created)')
			->order(['DATE(created)' => 'ASC'])
			->execute()
			->fetchAll('assoc');
	}

	public function ifExist(array $data): array
	{
		return $this->getConnection()
			->newQuery()
			->select('*')
			->from('video_watch')
			->where([
				'video_id' => $data['video_id'],
				'device_id' => $data['device_id']
			])
			->limit(1)
			->execute()
			->fetch('assoc') ?? [];
	}

	public function getAll(): array
	{
		return $this->find()
			->enableHydration(false)
			->all()
			->toList();
	}
}
