<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\ORM\Query;

class SoundsTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('sound');
		$this->setPrimaryKey('id');

		$this->belongsTo('SoundSection', [
			'foreignKey' => 'sound_section_id'
		]);

		$this->hasMany('Video', [
			'foreignKey' => 'sound_id',
			'dependent' => true
		]);

		$this->hasMany('SoundFavourite', [
			'foreignKey' => 'sound_id',
			'dependent' => true
		]);
	}

	public function getDetails(int $id): array
	{
		$result = $this->find()
			->where(['Sound.id' => $id])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function getSoundDetailsAgainstJobID(int $job_id): array
	{
		$result = $this->find()
			->select(['id', 'name', 'audio', 'description', 'sound_section_id', 'publish', 'created', 'modified'])
			->where(['Sound.job_id' => $job_id])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function getSoundsAgainstSection(int $sound_section_id, int $starting_point): array
	{
		return $this->find()
			->where(['Sound.sound_section_id' => $sound_section_id])
			->order(['Sound.id' => 'DESC'])
			->limit(10)
			->offset($starting_point * 10)
			->toArray();
	}

	public function getSearchSoundsAgainstSection(string $keyword, int $sound_section_id, int $starting_point): array
	{
		return $this->find()
			->where([
				'Sound.sound_section_id' => $sound_section_id,
				'Sound.publish' => 1,
				'OR' => [
					['Sound.name LIKE' => "%$keyword%"],
					['Sound.description LIKE' => "%$keyword%"]
				]
			])
			->order(['Sound.id' => 'DESC'])
			->limit(10)
			->offset($starting_point * 10)
			->toArray();
	}

	public function checkDuplicate(string $audio, string $created): int
	{
		return $this->find()
			->where([
				'Sound.audio' => $audio,
				'Sound.created' => $created
			])
			->count();
	}

	public function getSoundsCount(int $section_id): int
	{
		return $this->find()
			->where(['Sound.sound_section_id' => $section_id])
			->count();
	}

	public function getSoundDetailsAgainstAudio(string $audio): array
	{
		$result = $this->find()
			->where(['Sound.audio' => $audio])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function getFreqSounds(int $starting_id, array $freq_sounds): array
	{
		return $this->find()
			->where([
				'Sound.id IN' => $freq_sounds,
				'Sound.publish' => 1
			])
			->order(['Sound.name' => 'ASC'])
			->limit(10)
			->offset($starting_id * 10)
			->toArray();
	}

	public function getSounds(): array
	{
		return $this->find()
			->order(['Sound.id' => 'DESC'])
			->toArray();
	}

	public function getSearchResults(string $keyword, int $starting_point): array
	{
		return $this->find()
			->where([
				'Sound.publish' => 1,
				'OR' => [
					['Sound.name LIKE' => "%$keyword%"],
					['Sound.description LIKE' => "%$keyword%"]
				]
			])
			->limit(10)
			->offset($starting_point * 10)
			->toArray();
	}

	public function getSoundsAccordingToStatus(int $publish): array
	{
		return $this->find()
			->where(['Sound.publish' => $publish])
			->toArray();
	}

	public function getGlobalSounds(int $starting_id): array
	{
		return $this->find()
			->where(['Sound.publish' => 1])
			->order(['Sound.name' => 'ASC'])
			->limit(10)
			->offset($starting_id * 10)
			->toArray();
	}
}
?>
