<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;

class ProfileVisitsTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('profile_visit');
		$this->setPrimaryKey('id');

		$this->belongsTo('User', [
			'foreignKey' => 'user_id'
		]);

		$this->belongsTo('Visitor', [
			'className' => 'User',
			'foreignKey' => 'visitor_id'
		]);
	}

	public function getDetails(int $id): array
	{
		$result = $this->find()
			->where(['ProfileVisit.id' => $id])
			->first();

		return $result ? $result->toArray() : [];
	}

	public function getProfileViewsBetweenDates(int $user_id, string $start_datetime, string $end_datetime): int
	{
		return $this->find()
			->where([
				'ProfileVisit.user_id' => $user_id,
				'DATE(ProfileVisit.created) >=' => $start_datetime,
				'DATE(ProfileVisit.created) <=' => $end_datetime
			])
			->count();
	}

	public function getProfileVisitors(int $user_id, string $date, int $starting_point): array
	{
		return $this->find()
			->contain(['Visitor'])
			->select([
				'ProfileVisit.id', 'ProfileVisit.user_id', 'ProfileVisit.visitor_id', 'ProfileVisit.created',
				'Visitor.id', 'Visitor.first_name', 'Visitor.last_name', 'Visitor.bio', 'Visitor.website',
				'Visitor.profile_pic', 'Visitor.profile_pic_small', 'Visitor.profile_gif', 'Visitor.business',
				'Visitor.parent', 'Visitor.username', 'Visitor.device_token', 'Visitor.verified',
				'Visitor.created', 'Visitor.register_with'
			])
			->where([
				'ProfileVisit.user_id' => $user_id,
				'DATE(ProfileVisit.created) >' => $date,
				'ProfileVisit.visitor_id !=' => $user_id,
				'Visitor.id >' => 0
			])
			->group(['ProfileVisit.visitor_id'])
			->limit(10)
			->offset($starting_point * 10)
			->toArray();
	}

	public function getProfileVisitorsUnreadCount(int $user_id): int
	{
		$query = $this->find()
			->where([
				'ProfileVisit.user_id' => $user_id,
				'ProfileVisit.read' => 0
			])
			->group(['ProfileVisit.visitor_id']);

		return count($query->all());
	}

	public function updateReadCount(int $user_id): void
	{
		$this->updateAll(
			['read' => 1],
			['user_id' => $user_id]
		);
	}
}
?>
