<?php
namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\ORM\Query;

class HashtagVideosTable extends Table
{
	protected array $customFields = [
		"Video.id",
		"Video.user_id",
		"Video.description",
		"Video.video",
		"Video.thum",
		"Video.thum_small",
		"Video.gif",
		"Video.view",
		"Video.sound_id",
		"Video.privacy_type",
		"Video.allow_comments",
		"Video.allow_duet",
		"Video.duration",
		"Video.promote",
		"Video.pin_comment_id",
		"Video.pin",
		"Video.location_string",
		"Video.location_id",
		"Video.lat",
		"Video.long",
		"Video.width",
		"Video.height",
		"Video.user_thumbnail",
		"Video.default_thumbnail",
		"HashtagVideo.*",
		"Hashtag.*",
	];

	public function initialize(array $config): void
	{
		parent::initialize($config);
		$this->setTable('hashtag_video');
		$this->belongsTo('Hashtag', ['foreignKey' => 'hashtag_id']);
		$this->belongsTo('Video', ['foreignKey' => 'video_id']);
	}

	public function getDetails(int $id): array
	{
		$entity = $this->get($id);
		return $entity->toArray();
	}

	public function getHashtagVideos(int $hashtag_id): array
	{
		return $this->find()
			->contain([
				'Video.Sound',
				'Video.User.PrivacySetting',
				'Video.User.PushNotification',
				'Video.Location',
				'Hashtag',
			])
			->where([
				'HashtagVideo.hashtag_id'   => $hashtag_id,
				'Video.privacy_type'        => 'public',
				'Video.user_id >'           => 0,
			])
			->order(['Video.view' => 'DESC'])
			->all()
			->toArray();
	}

	public function getHashtagVideosLimit(int $hashtag_id): array
	{
		return $this->find()
			->contain(['Video', 'Hashtag'])
			->where([
				'HashtagVideo.hashtag_id' => $hashtag_id,
				'Video.privacy_type'      => 'public',
				'Video.user_id >'         => 0,
			])
			->order(['Video.view' => 'DESC'])
			->limit(5)
			->all()
			->toArray();
	}

	public function getHashtagVideosWithLimit(int $hashtag_id, int $starting_point): array
	{
		return $this->find()
			->contain(['Hashtag', 'Video'])
			->select($this->customFields)
			->where([
				'HashtagVideo.hashtag_id' => $hashtag_id,
				'Video.privacy_type'      => 'public',
			])
			->order(['Video.view' => 'DESC'])
			->limit(APP_RECORDS_PER_PAGE)
			->offset($starting_point * APP_RECORDS_PER_PAGE)
			->all()
			->toArray();
	}

	public function getHashtagVideosWithLimittemp(int $hashtag_id, int $starting_point): array
	{
		$fields = [
			"Video.id",
			"Video.user_id",
			"Video.description",
			"Video.video",
			"Video.thum",
			"Video.thum_small",
			"Video.gif",
			"Video.view",
			"Video.sound_id",
			"Video.privacy_type",
			"Video.allow_comments",
			"Video.allow_duet",
			"Video.duration",
			"Video.promote",
			"Video.pin_comment_id",
			"Video.pin",
			"Video.location_string",
			"Video.location_id",
			"Video.lat",
			"Video.long",
			"Video.width",
			"Video.height",
			"Video.user_thumbnail",
			"Video.default_thumbnail",
			"HashtagVideo.*",
		];

		return $this->find()
			->contain(['Hashtag', 'Video'])
			->select($fields)
			->where([
				'HashtagVideo.hashtag_id' => $hashtag_id,
				'Video.privacy_type'      => 'public',
				'Video.user_id >'         => 0,
			])
			->order(['Video.view' => 'DESC'])
			->limit(APP_RECORDS_PER_PAGE)
			->offset($starting_point * APP_RECORDS_PER_PAGE)
			->all()
			->toArray();
	}

	public function countHashtagViews(int $hashtag_id): array
	{
		$row = $this->find()
			->select(['total_sum' => $this->find()->func()->sum('Video.view')])
			->where([
				'HashtagVideo.hashtag_id' => $hashtag_id,
				'Video.privacy_type'      => 'public',
			])
			->first();

		return ['total_sum' => (int)$row->get('total_sum')];
	}

	public function countHashtagVideos(int $hashtag_id): int
	{
		return (int)$this->find()
			->where([
				'HashtagVideo.hashtag_id' => $hashtag_id,
				'Video.privacy_type'      => 'public',
			])
			->count();
	}

	public function getHastagsWhichHasGreaterNoOfVideos(int $starting_point): array
	{
		return $this->find()
			->select([
				'HashtagVideo.hashtag_id',
				'total_views' => $this->find()->func()->sum('Video.view'),
			])
			->contain(['Hashtag'])
			->leftJoinWith('Video')
			->group(['HashtagVideo.hashtag_id'])
			->order(['COUNT(HashtagVideo.hashtag_id)' => 'DESC'])
			->limit(5)
			->offset($starting_point * 5)
			->enableHydration(false)
			->toArray();
	}

	public function getHastagsWhichHasGreaterNoOfVideosAdmin(): array
	{
		return $this->find()
			->select([
				'HashtagVideo.hashtag_id',
				'total_views' => $this->find()->func()->sum('Video.view'),
			])
			->contain(['Hashtag'])
			->leftJoinWith('Video')
			->group(['HashtagVideo.hashtag_id'])
			->order(['total_views' => 'DESC'])
			->enableHydration(false)
			->toArray();
	}

	public function ifExist(array $data): array
	{
		$entity = $this->find()
			->where([
				'HashtagVideo.hashtag_id' => $data['hashtag_id'],
				'HashtagVideo.video_id'   => $data['video_id'],
			])
			->first();

		return $entity ? $entity->toArray() : [];
	}

	public function deleteDataAgainstVideo(int $video_id): bool
	{
		return (bool)$this->deleteAll(['HashtagVideo.video_id' => $video_id]);
	}
}
