<?php

declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;

class GiftSendsTable extends Table
{
	public function initialize(array $config): void
	{
		parent::initialize($config);

		$this->setTable('gift_send');
		$this->setPrimaryKey('id');

		$this->belongsTo('Gifts', [
			'foreignKey' => 'gift_id',
			'joinType' => 'INNER',
		]);

		$this->belongsTo('Users', [
			'foreignKey' => 'sender_id',
			'joinType' => 'INNER',
		]);
	}

	public function getDetails(int $id): array
	{
		$record = $this->find()
			->where(['id' => $id])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function countGiftSendByUser(int $user_id): array
	{
		$record = $this->find()
			->select(['total_amount' => $this->find()->func()->sum('coin')])
			->where(['sender_id' => $user_id])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function getDailySenders(string $date, int $starting_point): array
	{
		return $this->find()
			->select([
				'max_coin' => $this->find()->func()->max('coin'),
				'Users.id',
				'GiftSends.id',
				'GiftSends.sender_id',
				'GiftSends.coin'
			])
			->contain(['Users'])
			->where(["DATE(GiftSends.created) =" => $date])
			->group('GiftSends.sender_id')
			->orderDesc('max_coin')
			->limit(10)
			->offset($starting_point * 10)
			->all()
			->toList();
	}

	public function getDailyReceivers(string $date, int $starting_point): array
	{
		return $this->find()
			->select([
				'max_coin' => $this->find()->func()->max('coin'),
				'Users.id',
				'GiftSends.id',
				'GiftSends.receiver_id',
				'GiftSends.coin'
			])
			->contain(['Users'])
			->where(["DATE(GiftSends.created) =" => $date])
			->group('GiftSends.receiver_id')
			->orderDesc('max_coin')
			->limit(10)
			->offset($starting_point * 10)
			->all()
			->toList();
	}

	public function getHourlySenders(int $hour, int $starting_point): array
	{
		return $this->find()
			->select([
				'max_coin' => $this->find()->func()->max('coin'),
				'Users.id',
				'GiftSends.id',
				'GiftSends.sender_id',
				'GiftSends.coin'
			])
			->contain(['Users'])
			->where(["HOUR(GiftSends.created) =" => $hour])
			->group('GiftSends.sender_id')
			->orderDesc('max_coin')
			->limit(10)
			->offset($starting_point * 10)
			->all()
			->toList();
	}

	public function getHourlyReceivers(int $hour, int $starting_point): array
	{
		return $this->find()
			->select([
				'max_coin' => $this->find()->func()->max('coin'),
				'Users.id',
				'GiftSends.id',
				'GiftSends.receiver_id',
				'GiftSends.coin'
			])
			->contain(['Users'])
			->where(["HOUR(GiftSends.created) =" => $hour])
			->group('GiftSends.receiver_id')
			->orderDesc('max_coin')
			->limit(10)
			->offset($starting_point * 10)
			->all()
			->toList();
	}

	public function getTopGiftsSendByUser(int $user_id): array
	{
		return $this->find()
			->select([
				'Gift.id',
				'Gift.name',
				'Gift.image',
				'total' => $this->find()->func()->count('*')
			])
			->contain(['Gifts'])
			->where(['GiftSends.sender_id' => $user_id])
			->group('GiftSends.gift_id')
			->orderDesc('total')
			->limit(2)
			->all()
			->toList();
	}

	public function countGiftReceiveByUser(int $user_id): array
	{
		$record = $this->find()
			->select(['total_amount' => $this->find()->func()->sum('coin')])
			->where(['receiver_id' => $user_id])
			->first();

		return $record ? $record->toArray() : [];
	}

	public function getAll(): array
	{
		return $this->find()->all()->toList();
	}
}
