<?php
declare(strict_types=1);

namespace App\Lib;

use Kreait\Firebase\Factory;
use Kreait\Firebase\Auth;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Exception\MessagingException;
use Kreait\Firebase\Exception\Auth\UserNotFound;
use Kreait\Firebase\Exception\Auth\AuthError;
use Cake\Datasource\ConnectionManager;

class Firebase
{
    private static ?self $instance = null;
    private Auth $auth;
    private \Kreait\Firebase\Messaging $messaging;

    private function __construct()
    {
        $jsonFile = ROOT . '/config/' . FIREBASE_JSON_FILE;
        $factory = (new Factory)->withServiceAccount($jsonFile);

        $this->auth = $factory->createAuth();
        $this->messaging = $factory->createMessaging();
    }

    public static function getInstance(): self
    {
        return self::$instance ??= new self();
    }

    public function sendPushNotification(array $notification): bool
    {
        try {
            $message = CloudMessage::fromArray([
                'token' => $notification['to'],
                'notification' => $notification['notification'],
                'data' => $notification['data']
            ]);
            $this->messaging->send($message);
            return true;
        } catch (MessagingException|\Throwable $e) {
            return false;
        }
    }

    public function getFirebaseInfoAgainstUID(string $fuid): string|false
    {
        try {
            $user = $this->auth->getUser($fuid);
            return $user->providerData[0]->providerId ?? '';
        } catch (UserNotFound) {
            return false;
        }
    }

    public function postVideoToFirestore(array $videoData): bool
    {
        try {
            $conn = ConnectionManager::get('default');
            $conn->insert('video_summaries', [
                'video_id' => $videoData['video_id'],
                'data' => json_encode($videoData),
                'created_at' => date('Y-m-d H:i:s')
            ]);
            return true;
        } catch (\Throwable) {
            return false;
        }
    }

    public function deleteVideoFromFirestore(int $videoId): bool
    {
        try {
            $conn = ConnectionManager::get('default');
            $conn->delete('video_summaries', ['video_id' => $videoId]);
            return true;
        } catch (\Throwable) {
            return false;
        }
    }

    public function testEmail(string $email): ?string
    {
        try {
            $this->auth->sendEmailVerificationLink($email);
            return null;
        } catch (\Throwable $e) {
            return $e->getMessage();
        }
    }

    public function getUserDetailsAgainstEmail(string $email): object|false
    {
        try {
            return $this->auth->getUserByEmail($email);
        } catch (UserNotFound) {
            return false;
        }
    }

    public function getUserDetailsAgainstUID(string $uid): object|false
    {
        try {
            return $this->auth->getUser($uid);
        } catch (UserNotFound) {
            return false;
        }
    }

    public function createUser(array $data): object
    {
        return $this->auth->createUser([
            'email' => $data['email'],
            'emailVerified' => false,
            'password' => $data['password'] ?? '123456',
            'displayName' => trim(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? '')),
            'disabled' => false
        ]);
    }

    public function deleteUser(string $uid): bool
    {
        try {
            $this->auth->deleteUser($uid);
            return true;
        } catch (UserNotFound) {
            return false;
        } catch (AuthError) {
            return true;
        }
    }

    public function changePassword(string $uid, string $password): bool
    {
        try {
            $this->auth->changeUserPassword($uid, $password);
            return true;
        } catch (UserNotFound) {
            return false;
        } catch (AuthError) {
            return true;
        }
    }
}