<?php
declare(strict_types=1);

namespace App\Controller;

use App\Lib\Utility;
use App\Lib\Message;
use App\Lib\CustomEmail;
use App\Lib\Mealme;
use App\Lib\Firebase;
use App\Lib\Aws;
use Cake\Event\EventInterface;
use Cake\Http\Exception\BadRequestException;

class AdminController extends AppController
{
	protected Firebase $firebase;
	protected Aws $aws;

	public function initialize(): void
	{
		parent::initialize();
		$this->firebase = Firebase::getInstance();
		$this->aws = Aws::getInstance();
		$this->loadComponent('RequestHandler');
	}

	public function beforeFilter(EventInterface $event): void
	{
		parent::beforeFilter($event);
		$json = file_get_contents('php://input');
		$jsonError = Utility::isJsonError($json);
		if ($jsonError === 'false') {
			$clientApiKey = $this->getRequest()->getHeaderLine('Api-Key') ?: $this->getRequest()->getHeaderLine('API-KEY') ?: $this->getRequest()->getHeaderLine('api-key');
			if (strlen($clientApiKey) > 0) {
				if ($clientApiKey !== ADMIN_API_KEY) {
					Message::ACCESSRESTRICTED();
					die();
				}
			} else {
				echo json_encode(['code' => 201, 'msg' => 'API KEY is missing']);
				die();
			}
		} else {
			echo json_encode(['code' => 201, 'msg' => $jsonError]);
			die();
		}
	}

	public function index(): void
	{
		echo 'Congratulations!. You have configured your mobile api correctly';
	}

	public function login(): void
	{
		$this->loadModel('Admin');
		if ($this->getRequest()->is('post')) {
			$json = file_get_contents('php://input');
			$data = json_decode($json, true);
			$email = strtolower($data['email'] ?? '');
			$password = $data['password'] ?? null;
			if ($email !== '' && $password !== null) {
				$userData = $this->Admin->loginAllUsers($email, $password);
				if ($userData) {
					$userId = $userData[0]['Admin']['id'];
					$userDetails = $this->Admin->getUserDetailsFromID($userId);
					echo json_encode(['code' => 200, 'msg' => $userDetails]);
				} else {
					echo Message::INVALIDDETAILS();
					die();
				}
			} else {
				echo Message::ERROR();
				die();
			}
		} else {
			throw new BadRequestException();
		}
	}

	public function importMealmeStores(): void
	{
		$json = file_get_contents('php://input');
		$data = json_decode($json, true);
		$this->loadModel('Store');
		foreach ($data as $d) {
			pr($d);
			die();
		}
	}

	public function addCoupon(): void
	{
		if (APP_STATUS === 'demo') {
			echo json_encode(['code' => 201, 'msg' => 'Sorry this feature is disabled in demo mode']);
			die();
		}
		$this->loadModel('Coupon');
		if ($this->getRequest()->is('post')) {
			$json = file_get_contents('php://input');
			$data = json_decode($json, true);
			$couponCode = $data['coupon_code'] ?? null;
			$limitUsers = $data['limit_users'] ?? null;
			$discount = $data['discount'] ?? null;
			$expiryDate = $data['expiry_date'] ?? null;
			$coupon = [
				'coupon_code' => $couponCode,
				'limit_users' => $limitUsers,
				'discount' => $discount,
				'expiry_date' => $expiryDate,
			];
			if (isset($data['id'])) {
				$this->Coupon->id = $data['id'];
				$this->Coupon->save($coupon);
				$couponDetail = $this->Coupon->getDetails($data['id']);
				echo json_encode(['code' => 200, 'msg' => $couponDetail]);
				die();
			}
			if (empty($this->Coupon->isCouponCodeExist($couponCode))) {
				if ($this->Coupon->save($coupon)) {
					$couponDetail = $this->Coupon->getDetails($this->Coupon->id);
					echo json_encode(['code' => 200, 'msg' => $couponDetail]);
					die();
				}
				echo Message::DATASAVEERROR();
				die();
			}
			Message::DUPLICATEDATE();
			die();
		} else {
			throw new BadRequestException();
		}
	}

	public function showBusinessSubmissions(): void
	{
		$this->loadModel('Business');
		if ($this->getRequest()->is('post')) {
			$json = file_get_contents('php://input');
			$data = json_decode($json, true);
			$details = isset($data['business_id']) ? $this->Business->getDetails($data['business_id']) : $this->Business->getAll();
			if (!empty($details)) {
				echo json_encode(['code' => 200, 'msg' => $details]);
				die();
			}
			Message::EMPTYDATA();
			die();
		} else {
			throw new BadRequestException();
		}
	}

	public function showDocuments(): void
	{
		$this->loadModel('Document');
		if ($this->getRequest()->is('post')) {
			$json = file_get_contents('php://input');
			$data = json_decode($json, true);
			$details = $this->Document->getDocument($data['user_id'] ?? null);
			if (!empty($details)) {
				echo json_encode(['code' => 200, 'msg' => $details]);
				die();
			}
			Message::EMPTYDATA();
			die();
		} else {
			throw new BadRequestException();
		}
	}

	public function showCoupons(): void
	{
		$this->loadModel('Coupon');
		if ($this->getRequest()->is('post')) {
			$json = file_get_contents('php://input');
			$data = json_decode($json, true);
			$coupons = isset($data['coupon_id']) ? $this->Coupon->getDetails($data['coupon_id']) : $this->Coupon->getAll();
			if (!empty($coupons)) {
				echo json_encode(['code' => 200, 'msg' => $coupons]);
				die();
			}
			Message::EMPTYDATA();
			die();
		} else {
			throw new BadRequestException();
		}
	}

	public function deleteCoupon(): void
	{
		if (Configure::read('App.status') === 'demo') {
			$this->set([
				'code' => 201,
				'msg' => 'Sorry this feature is disabled in demo mode',
				'_serialize' => ['code', 'msg']
			]);
			return;
		}

		$this->loadModel('Coupons');

		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$couponId = $data['coupon_id'] ?? null;

			if ($couponId === null) {
				$this->set([
					'code' => 400,
					'msg' => 'Coupon ID is required',
					'_serialize' => ['code', 'msg']
				]);
				return;
			}

			$coupon = $this->Coupons->find()
				->where(['id' => $couponId])
				->first();

			if ($coupon) {
				if ($this->Coupons->delete($coupon)) {
					$this->set([
						'code' => 200,
						'msg' => 'Deleted successfully',
						'_serialize' => ['code', 'msg']
					]);
				} else {
					$this->set([
						'code' => 500,
						'msg' => 'Data save error',
						'_serialize' => ['code', 'msg']
					]);
				}
			} else {
				$this->set([
					'code' => 404,
					'msg' => 'No data found',
					'_serialize' => ['code', 'msg']
				]);
			}
		} else {
			$this->set([
				'code' => 405,
				'msg' => 'Invalid request method',
				'_serialize' => ['code', 'msg']
			]);
		}
	}

	public function showUsers(): void
	{
		$this->loadModel('Users');

		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$role = $data['role'] ?? 'user';
			$start = isset($data['starting_point']) && is_numeric($data['starting_point']) ? (int)$data['starting_point'] : 0;

			$users = $this->Users->getUsers($start);

			if (empty($users)) {
				$output = [
					'code' => 404,
					'msg' => 'No data found'
				];
			} else {
				foreach ($users as $i => $user) {
					$walletTotal = $this->walletTotal($user->id);
					$user->wallet = $walletTotal['total'] ?? 0;
				}

				$count = $this->Users->totalCountGetUsers($role);

				$output = [
					'code' => 200,
					'msg' => $users,
					'total_pages' => (int)ceil($count / (defined('ADMIN_RECORDS_PER_PAGE') ? ADMIN_RECORDS_PER_PAGE : 10))
				];
			}

			$this->set([
				'code' => $output['code'],
				'msg' => $output['msg'],
				'total_pages' => $output['total_pages'] ?? null,
				'_serialize' => ['code', 'msg', 'total_pages']
			]);
		}
	}

	public function postVideo(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('Videos');
		$this->loadModel('Sounds');
		$this->loadModel('Hashtags');
		$this->loadModel('HashtagVideos');
		$this->loadModel('Users');
		$this->loadModel('Notifications');
		$this->loadModel('Followers');
		$this->loadModel('PushNotifications');

		$request = $this->getRequest();
		if ($request->is('post')) {
			$ffmpeg = new \App\Utility\Ffmpeg();
			$created = date('Y-m-d H:i:s');
			$data = $request->getData();
			$userId = $data['user_id'] ?? 0;
			$user = $this->Users->getUserDetailsFromID((int)$userId);
			if (empty($user)) {
				return $this->getResponse()
					->withType('application/json')
					->withStringBody(json_encode(['code' => 404, 'msg' => 'User not found']));
			}
			$soundId = $data['sound_id'] ?? 0;
			$videoId = $data['video_id'] ?? 0;
			$sound = $this->Sounds->getDetails((int)$soundId);
			$orig = [];
			if ($videoId) {
				$orig = $this->Videos->getDetails((int)$videoId);
				$sound = $this->Sounds->getDetails((int)$orig['Video']['sound_id']);
			}
			if (defined('MEDIA_STORAGE') && MEDIA_STORAGE === 's3' && method_exists($this->Aws, 's3_video_upload')) {
				$res = $this->Aws->s3_video_upload($user['User']['id'], 'video', $sound, $orig, (int)($data['duet'] ?? 0));
				$video = \App\Utility\Utility::getCloudFrontUrl($res['video'], '/video');
				$gif = \App\Utility\Utility::getCloudFrontUrl($res['gif'], '/gif');
				$thum = \App\Utility\Utility::getCloudFrontUrl($res['thum'], '/thum');
				$nudity = $res['video'];
			} else {
				$res = \App\Utility\Regular::local_video_upload($user['User']['id'], 'video', $sound, $orig, (int)($data['duet'] ?? 0));
				$video = $res['video'];
				$gif = $res['gif'];
				$thum = $res['thum'];
				$nudity = BASE_URL . $res['video'];
			}
			$entity = $this->Videos->newEmptyEntity();
			$entity = $this->Videos->patchEntity($entity, [
				'user_id' => $user['User']['id'],
				'sound_id' => $soundId,
				'gif' => $gif,
				'duration' => $ffmpeg->getDurationOfVideoFile($video),
				'video' => $video,
				'thum' => $thum,
				'description' => $data['description'] ?? '',
				'privacy_type' => strtolower($data['privacy_type'] ?? ''),
				'allow_comments' => (int)($data['allow_comments'] ?? 0),
				'allow_duet' => (int)($data['allow_duet'] ?? 0),
				'interest_id' => $data['interest_id'] ?? null,
				'lang_id' => $data['lang_id'] ?? null,
				'story' => $data['story'] ?? '',
				'product_id' => $data['product_id'] ?? null,
				'created' => $created
			]);
			if (!$this->Videos->save($entity)) {
				return $this->getResponse()
					->withType('application/json')
					->withStringBody(json_encode(['code' => 500, 'msg' => 'Could not save video']));
			}
			if (!empty(DEEPENGIN_KEY)) {
				\App\Utility\Utility::checkNudity($nudity, $entity->id);
			}
			$hashtags = json_decode($data['hashtags_json'] ?? '[]', true);
			if (is_array($hashtags)) {
				foreach ($hashtags as $tag) {
					$name = strtolower($tag['name']);
					$exist = $this->Hashtags->ifExist($name);
					$hashtag = $exist['Hashtag'] ?? null;
					if ($hashtag) {
						$hId = $hashtag['id'];
					} else {
						$newTag = $this->Hashtags->newEmptyEntity();
						$newTag = $this->Hashtags->patchEntity($newTag, ['name' => $name, 'lang_id' => $data['lang_id'] ?? null]);
						$savedTag = $this->Hashtags->save($newTag);
						$hId = $savedTag->id ?? null;
					}
					if ($hId) {
						$tagVideo = $this->HashtagVideos->newEmptyEntity();
						$tagVideo = $this->HashtagVideos->patchEntity($tagVideo, ['video_id' => $entity->id, 'hashtag_id' => $hId]);
						$this->HashtagVideos->save($tagVideo);
					}
				}
			}
			$usersTagged = json_decode($data['users_json'] ?? '[]', true);
			if (is_array($usersTagged)) {
				foreach ($usersTagged as $tag) {
					$tUser = $this->Users->getUserDetailsFromID((int)$tag['user_id']);
					if (!empty($tUser['User']['device_token'])) {
						$msg = $user['User']['username'] . ' tagged you';
						$cfg = $this->PushNotifications->getDetails($tUser['User']['id']);
						if (!empty($cfg['PushNotification']['video_updates'])) {
							$this->firebase->sendPushNotification([
								'to' => $tUser['User']['device_token'],
								'notification' => ['title' => $msg, 'type' => 'video_tag'],
								'data' => ['type' => 'video_tag', 'receiver_id' => $tUser['User']['id']]
							]);
						}
						$notification = $this->Notifications->newEmptyEntity();
						$notification = $this->Notifications->patchEntity($notification, [
							'sender_id' => $user['User']['id'],
							'receiver_id' => $tUser['User']['id'],
							'type' => 'video_tag',
							'video_id' => $entity->id,
							'string' => $msg,
							'created' => $created
						]);
						$this->Notifications->save($notification);
					}
				}
			}
			$followers = $this->Followers->getUserFollowersWithoutLimit($user['User']['id']);
			foreach ($followers as $follower) {
				$fUser = $follower['FollowerList'] ?? null;
				$token = $fUser['device_token'] ?? null;
				if ($fUser && $token) {
					$msg = $user['User']['username'] . ' posted a video';
					$cfg = $this->PushNotifications->getDetails($fUser['id']);
					if (!empty($cfg['PushNotification']['video_updates'])) {
						$this->firebase->sendPushNotification([
							'to' => $token,
							'notification' => ['title' => $msg, 'type' => 'video_updates', 'video_id' => $entity->id],
							'data' => ['type' => 'video_updates', 'receiver_id' => $fUser['id']]
						]);
					}
					$notification = $this->Notifications->newEmptyEntity();
					$notification = $this->Notifications->patchEntity($notification, [
						'sender_id' => $user['User']['id'],
						'receiver_id' => $fUser['id'],
						'type' => 'video_updates',
						'video_id' => $entity->id,
						'string' => $msg,
						'created' => $created
					]);
					$this->Notifications->save($notification);
				}
			}
			$details = $this->Videos->getDetails($entity->id);
			return $this->getResponse()
				->withType('application/json')
				->withStringBody(json_encode(['code' => 200, 'msg' => $details]));
		}
		return $this->getResponse()
			->withType('application/json')
			->withStringBody(json_encode(['code' => 405, 'msg' => 'Method Not Allowed']));
	}

	public function purchaseCoin(): Response
	{
		$this->loadModel('Users');
		$this->loadModel('PurchaseCoins');

		if ($this->request->is('post')) {
			$data = $this->request->getData();

			$userId = $data['user_id'] ?? 0;
			$user = $this->Users->getUserDetailsFromID($userId);

			if (empty($user)) {
				$response = ['code' => 404, 'msg' => 'User not found'];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($response));
			}

			$entityData = [
				'user_id'        => $userId,
				'title'          => $data['title'] ?? '',
				'coin'           => $data['coin'] ?? 0,
				'transaction_id' => $data['transaction_id'] ?? '',
				'device'         => $data['device'] ?? '',
				'created'        => FrozenTime::now()
			];

			$entity = $this->PurchaseCoins->patchEntity($this->PurchaseCoins->newEmptyEntity(), $entityData);
			$this->PurchaseCoins->saveOrFail($entity);

			$details = $this->PurchaseCoins->getDetails($entity->id);
			$details['User']['wallet'] = $this->walletTotal($user['User']['id'])['total'] ?? 0;

			$response = ['code' => 200, 'msg' => $details];
			return $this->response
				->withType('application/json')
				->withStringBody(json_encode($response));
		}

		$response = ['code' => 405, 'msg' => 'Invalid request method'];
		return $this->response
			->withType('application/json')
			->withStringBody(json_encode($response));
	}

	public function walletTotal(string $user_id): array
	{
		$withdrawRequestTable = $this->getTableLocator()->get('WithdrawRequest');
		$promotionTable = $this->getTableLocator()->get('Promotion');
		$liveStreamingTable = $this->getTableLocator()->get('LiveStreaming');
		$liveStreamingWatchTable = $this->getTableLocator()->get('LiveStreamingWatch');
		$referralUsedTable = $this->getTableLocator()->get('ReferralUsed');
		$settingTable = $this->getTableLocator()->get('Setting');
		$purchaseAmountTotal = [['total_amount' => 0]];
		$giftReceive = [['total_amount' => 0]];
		$giftSend = [['total_amount' => 0]];
		$withdrawRequestDetail = $withdrawRequestTable->getTotalCoins($user_id);
		if (empty($withdrawRequestDetail[0]['total_coin'])) {
			$withdrawRequestDetail[0]['total_coin'] = 0;
		}
		if (in_array('Gift', $this->getTableLocator()->listTables())) {
			$purchaseCoinTable = $this->getTableLocator()->get('PurchaseCoin');
			$giftSendTable = $this->getTableLocator()->get('GiftSend');
			$purchaseAmountTotal = $purchaseCoinTable->totalAmountPurchase($user_id);
			$giftSend = $giftSendTable->countGiftSendByUser($user_id);
			$giftReceive = $giftSendTable->countGiftReceiveByUser($user_id);
		}
		$referralCount = $referralUsedTable->countReferralUsedByOthers($user_id);
		$referralValue = 0;
		$settingEntity = $settingTable->find()->where(['key' => 'referral_coin'])->first();
		if ($settingEntity) {
			$referralValue = $settingEntity->value;
		}
		$referralEarn = $referralCount * $referralValue;
		$promotionCoin = $promotionTable->countPromotionCoin($user_id);
		$liveStreamEarnedCoin = $liveStreamingTable->countCoinsEarnedByUser($user_id);
		$liveStreamWatchCoin = $liveStreamingWatchTable->countCoinsSpendByUser($user_id);
		$giftReceive[0]['total_amount'] = $giftReceive[0]['total_amount'] ?: 0;
		$giftSend[0]['total_amount'] = $giftSend[0]['total_amount'] ?: 0;
		$promotionCoin[0]['total_amount'] = $promotionCoin[0]['total_amount'] ?: 0;
		$liveStreamEarnedCoin[0]['total_amount'] = $liveStreamEarnedCoin[0]['total_amount'] ?: 0;
		$liveStreamWatchCoin[0]['total_amount'] = $liveStreamWatchCoin[0]['total_amount'] ?: 0;
		$purchaseAmount = $purchaseAmountTotal[0]['total_amount'] ?? 0;
		$earnedMoney = $giftReceive[0]['total_amount'] + $purchaseAmount + $liveStreamEarnedCoin[0]['total_amount'] + $referralEarn;
		$totalMoney = $earnedMoney - $giftSend[0]['total_amount'] - $promotionCoin[0]['total_amount'] - $liveStreamWatchCoin[0]['total_amount'] - $withdrawRequestDetail[0]['total_coin'];
		return [
			'gifts_receive' => (string)$giftReceive[0]['total_amount'],
			'gifts_send'    => (string)$giftSend[0]['total_amount'],
			'total'         => (string)$totalMoney
		];
	}

	public function showUsersInfo(): \Psr\Http\Message\ResponseInterface
	{
		$this->getRequest()->allowMethod(['post']);
		$users = (array)$this->getRequest()->getData('users');
		$ids = array_column($users, 'user_id');
		$userInfos = $this->getTableLocator()->get('Users')->getMultipleUsersData($ids);
		$output = ['code' => 200, 'msg' => $userInfos];
		return $this->getResponse()
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function sendPushNotificationToAllUsers(): \Psr\Http\Message\ResponseInterface
	{
		$this->getRequest()->allowMethod(['post']);
		$text = (string)$this->getRequest()->getData('text');
		$users = $this->getTableLocator()->get('Users')->getAllUsersNotification();
		foreach ($users as $user) {
			$token = $user['device_token'] ?? '';
			if (strlen($token) > 15) {
				$notification = [
					'to'           => $token,
					'notification' => [
						'title' => '',
						'body'  => $text,
						'badge' => '1',
						'sound' => 'default',
						'icon'  => ''
					],
					'data'         => [
						'title' => '',
						'body'  => $text,
						'icon'  => '',
						'badge' => '1',
						'sound' => 'default'
					]
				];
				$this->firebase->sendPushNotification($notification);
			}
		}
		$output = ['code' => 200, 'msg' => 'successfully sent'];
		return $this->getResponse()
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function deleteVideoComment(): \Psr\Http\Message\ResponseInterface
	{
		$this->getRequest()->allowMethod(['post']);
		$commentId = (int)$this->getRequest()->getData('comment_id');
		$table = $this->getTableLocator()->get('VideoComments');
		$comment = $table->find()->where(['id' => $commentId])->first();
		if ($comment) {
			$table->delete($comment);
			$output = ['code' => 200, 'msg' => 'deleted'];
		} else {
			$output = ['code' => 404, 'msg' => 'no data'];
		}
		return $this->getResponse()
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function showUserLikedVideos(): \Psr\Http\Message\ResponseInterface
	{
		$this->getRequest()->allowMethod(['post']);
		$userId = (int)$this->getRequest()->getData('user_id');
		$likeTable = $this->getTableLocator()->get('VideoLikes');
		$videos = $likeTable->getUserAllVideoLikes($userId);
		if (empty($videos)) {
			$output = ['code' => 204, 'msg' => []];
			return $this->getResponse()
				->withType('application/json')
				->withStringBody(json_encode($output));
		}
		foreach ($videos as &$video) {
			$video['Video']['like_count'] = $likeTable->countLikes($video['Video']['id']);
		}
		$output = ['code' => 200, 'msg' => $videos];
		return $this->getResponse()
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function showVideos(): \Psr\Http\Message\ResponseInterface
	{
		$request = $this->getRequest();
		$request->allowMethod(['post']);
		$data = (array)$request->getData();
		$videoTable   = $this->getTableLocator()->get('Video');
		$commentTable = $this->getTableLocator()->get('VideoComment');
		$likeTable    = $this->getTableLocator()->get('VideoLike');
		if (!empty($data['user_id'])) {
			$videos = $videoTable->getUserVideos((int)$data['user_id']);
		} else {
			$videos = $videoTable->getAllVideos((int)($data['starting_point'] ?? 0));
		}
		foreach ($videos as &$video) {
			$video['Video']['comment_count'] = $commentTable->countComments($video['Video']['id']);
			$video['Video']['like_count']    = $likeTable->countLikes($video['Video']['id']);
		}
		$response = $this->getResponse()->withType('application/json');
		return $response->withStringBody(json_encode(['code' => 200, 'msg' => $videos]));
	}

	public function showNudityDetectedVideos(): \Psr\Http\Message\ResponseInterface
	{
		$request = $this->getRequest();
		$request->allowMethod(['post']);
		$videoTable = $this->getTableLocator()->get('Video');
		$videos     = $videoTable->getAllNudityDetectedVideos((int)$request->getData('starting_point'));
		if (empty($videos)) {
			$response = $this->getResponse()->withType('application/json');
			return $response->withStringBody(json_encode(['code' => 204, 'msg' => []]));
		}
		$commentTable = $this->getTableLocator()->get('VideoComment');
		$likeTable    = $this->getTableLocator()->get('VideoLike');
		foreach ($videos as &$video) {
			$video['Video']['comment_count'] = $commentTable->countComments($video['Video']['id']);
			$video['Video']['like_count']    = $likeTable->countLikes($video['Video']['id']);
		}
		$response = $this->getResponse()->withType('application/json');
		return $response->withStringBody(json_encode(['code' => 200, 'msg' => $videos]));
	}

	public function getUserDetailsFromCache(int|string $user_id): mixed
	{
		$cacheKey = 'user_details_' . $user_id;
		$details = Cache::read($cacheKey, 'default');

		if ($details !== null) {
			return $details;
		}

		$userTable = $this->fetchTable('User');
		$details = $userTable->getObjectsForCache($user_id);

		Cache::write($cacheKey, $details, 'default');
		return $details;
	}

	public function showVideoDetail(): \Psr\Http\Message\ResponseInterface
	{
		$request    = $this->getRequest();
		$request->allowMethod(['post']);
		$videoTable = $this->getTableLocator()->get('Video');
		$videoId    = (int)$request->getData('video_id');
		$videoDetail = $videoTable->getDetailsAdmin($videoId);
		if (empty($videoDetail)) {
			$response = $this->getResponse()->withType('application/json');
			return $response->withStringBody(json_encode(['code' => 204, 'msg' => []]));
		}
		$videoDetail['User']                 = $this->getUserDetailsFromCache($videoDetail['Video']['user_id'])['User'];
		$videoDetail['Video']['like_count']    = $this->getTableLocator()->get('VideoLike')->countLikes($videoDetail['Video']['id']);
		$videoDetail['Video']['comment_count'] = $this->getTableLocator()->get('VideoComment')->countComments($videoDetail['Video']['id']);
		$response = $this->getResponse()->withType('application/json');
		return $response->withStringBody(json_encode(['code' => 200, 'msg' => $videoDetail]));
	}
    
	public function showAllVerificationRequests(): void
	{
		$this->loadModel('VerificationRequests');
		if ($this->getRequest()->is('post')) {
			$data = $this->getRequest()->getData();
			$details = $this->VerificationRequests->find()->toArray();
			if (!empty($details)) {
				$output = ['code' => 200, 'msg' => $details];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			} else {
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode(['code' => 204, 'msg' => 'No data found']));
				return;
			}
		}
	}

	public function showVideosAgainstHashtag(): void
	{
		$this->loadModel('HashtagVideos');
		$this->loadModel('Videos');
		$this->loadModel('VideoLikes');
		$this->loadModel('VideoFavourites');
		$this->loadModel('VideoComments');

		if ($this->getRequest()->is('post')) {
			$data = $this->getRequest()->getData();
			$hashtag_id = $data['hashtag_id'] ?? null;
			if ($hashtag_id === null) {
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode(['code' => 400, 'msg' => 'Hashtag ID is required']));
				return;
			}

			$videos = $this->HashtagVideos->getHashtagVideos($hashtag_id);

			if (!empty($videos)) {
				foreach ($videos as $key => $video) {
					$videoId = $video['Video']['id'];
					$videos[$key]['Video']['comment_count'] = $this->VideoComments->countComments($videoId);
					$videos[$key]['Video']['like_count'] = $this->VideoLikes->countLikes($videoId);
				}

				$output = ['code' => 200, 'msg' => $videos];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			} else {
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode(['code' => 204, 'msg' => 'No videos found']));
				return;
			}
		}
	}

	public function showVideosAgainstSound(): void
	{
		$this->loadModel('Sounds');
		$this->loadModel('Videos');
		$this->loadModel('VideoLikes');
		$this->loadModel('VideoFavourites');
		$this->loadModel('VideoComments');

		if ($this->getRequest()->is('post')) {
			$data = $this->getRequest()->getData();
			$user_id = $data['user_id'] ?? 0;
			$device_id = $data['device_id'] ?? null;
			$starting_point = $data['starting_point'] ?? 0;
			$sound_id = $data['sound_id'] ?? null;

			if ($device_id === null || $sound_id === null) {
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode(['code' => 400, 'msg' => 'Missing required parameters']));
				return;
			}

			$videos = $this->Videos->getVideosAgainstSoundID($user_id, $device_id, $starting_point, $sound_id);

			if (!empty($videos)) {
				foreach ($videos as $key => $video) {
					$videoId = $video['Video']['id'];
					$videos[$key]['Video']['comment_count'] = $this->VideoComments->countComments($videoId);
					$videos[$key]['Video']['like_count'] = $this->VideoLikes->countLikes($videoId);
				}

				$output = ['code' => 200, 'msg' => $videos];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			} else {
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode(['code' => 204, 'msg' => 'No videos found']));
				return;
			}
		}
	}
    
	public function updateVerificationRequest(): void
	{
		$this->loadModel('VerificationRequest');
		$this->loadModel('User');

		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$verified = $data['verified'] ?? null;
			$verificationRequestId = $data['verification_request_id'] ?? null;

			if ($verificationRequestId === null || $verified === null) {
				Message::EMPTYDATA();
				return;
			}

			$details = $this->VerificationRequest->getDetails($verificationRequestId);

			if (!empty($details)) {
				$userId = $details['VerificationRequest']['user_id'] ?? null;

				if ($userId !== null) {
					$user = $this->User->get($userId);
					$user->verified = $verified;
					$this->User->save($user);

					$verificationRequest = $this->VerificationRequest->get($verificationRequestId);
					$verificationRequest->verified = $verified;
					$this->VerificationRequest->save($verificationRequest);

					$updatedDetails = $this->User->getUserDetailsFromID($userId);

					$output = ['code' => 200, 'msg' => $updatedDetails];
					$this->response = $this->response->withType('application/json')
													 ->withStringBody(json_encode($output));
					return;
				}
			}

			Message::EMPTYDATA();
		}
	}

	public function deleteSoundSection(): void
	{
		$this->loadModel('SoundSection');

		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$id = $data['id'] ?? null;

			if ($id === null) {
				Message::EMPTYDATA();
				return;
			}

			$details = $this->SoundSection->getDetails($id);

			if (!empty($details)) {
				$entity = $this->SoundSection->get($id);
				$this->SoundSection->delete($entity);

				$output = ['code' => 200, 'msg' => 'deleted'];
				$this->response = $this->response->withType('application/json')
												 ->withStringBody(json_encode($output));
				return;
			}

			Message::EMPTYDATA();
		}
	}
    
	public function addSoundSection(): void
	{
		$this->autoRender = false;
		$this->request->allowMethod(['post']);
		$this->loadModel('SoundSection');
		$data = $this->request->getData();
		if (!empty($data['id'])) {
			$entity = $this->SoundSection->get($data['id']);
			$entity->name = $data['name'];
			$entity->created = date('Y-m-d H:i:s');
			if ($this->SoundSection->save($entity)) {
				$details = $this->SoundSection->get($data['id']);
				echo json_encode(['code' => 200, 'msg' => $details]);
			} else {
				http_response_code(400);
				echo json_encode(['code' => 400, 'msg' => 'Update failed']);
			}
			return;
		}
		$count = $this->SoundSection->find()->where(['name' => $data['name']])->count();
		if ($count === 0) {
			$entity = $this->SoundSection->newEntity(['name' => $data['name'], 'created' => date('Y-m-d H:i:s')]);
			if ($this->SoundSection->save($entity)) {
				$details = $this->SoundSection->get($entity->id);
				echo json_encode(['code' => 200, 'msg' => $details]);
			} else {
				http_response_code(400);
				echo json_encode(['code' => 400, 'msg' => 'Save failed']);
			}
		} else {
			echo Message::DUPLICATEDATE();
		}
	}

	public function addInterestSection(): void
	{
		$this->autoRender = false;
		$this->request->allowMethod(['post']);
		$this->loadModel('InterestSection');
		$data = $this->request->getData();
		if (!empty($data['id'])) {
			$entity = $this->InterestSection->get($data['id']);
			$entity->title = $data['title'];
			$entity->created = date('Y-m-d H:i:s');
			if ($this->InterestSection->save($entity)) {
				$details = $this->InterestSection->get($data['id']);
				echo json_encode(['code' => 200, 'msg' => $details]);
			} else {
				http_response_code(400);
				echo json_encode(['code' => 400, 'msg' => 'Update failed']);
			}
			return;
		}
		$count = $this->InterestSection->find()->where(['title' => $data['title']])->count();
		if ($count === 0) {
			$entity = $this->InterestSection->newEntity(['title' => $data['title'], 'created' => date('Y-m-d H:i:s')]);
			if ($this->InterestSection->save($entity)) {
				$details = $this->InterestSection->get($entity->id);
				echo json_encode(['code' => 200, 'msg' => $details]);
			} else {
				http_response_code(400);
				echo json_encode(['code' => 400, 'msg' => 'Save failed']);
			}
		} else {
			echo Message::DUPLICATEDATE();
		}
	}

	public function deleteInterestSection(): void
	{
		$this->autoRender = false;
		$this->request->allowMethod(['post']);
		$this->loadModel('InterestSection');
		$data = $this->request->getData();
		$id = $data['id'];
		try {
			$entity = $this->InterestSection->get($id);
			if ($this->InterestSection->delete($entity)) {
				echo json_encode(['code' => 200, 'msg' => 'deleted']);
			} else {
				http_response_code(400);
				echo json_encode(['code' => 400, 'msg' => 'Delete failed']);
			}
		} catch (\Cake\Datasource\Exception\RecordNotFoundException $e) {
			echo json_encode(['code' => 201, 'msg' => 'Invalid id: Do not exist']);
		}
	}

	public function showInterestSection(): void
	{
		$this->autoRender = false;
		$this->request->allowMethod(['post']);
		$this->loadModel('InterestSection');
		$details = $this->InterestSection->find()->all()->toArray();
		if (!empty($details)) {
			echo json_encode(['code' => 200, 'msg' => $details]);
		} else {
			echo Message::EMPTYDATA();
		}
	}

	public function showInterestAgainstSection(): void
	{
		$this->autoRender = false;
		$this->request->allowMethod(['post']);
		$this->loadModel('Interest');
		$data = $this->request->getData();
		$sectionId = $data['interest_section_id'];
		$details = $this->Interest->find()->where(['interest_section_id' => $sectionId])->all()->toArray();
		if (!empty($details)) {
			echo json_encode(['code' => 200, 'msg' => $details]);
		} else {
			echo Message::EMPTYDATA();
		}
	}
    
	public function sendVideoPushNotificationToAllUsers(): void
	{
		$this->loadModel('User');
		$this->loadModel('Video');
		if ($this->request->is('post')) {
			$data = $this->request->input('json_decode', true);
			$txt = $data['text'] ?? '';
			$video_id = $data['video_id'] ?? null;
			$title = $data['title'] ?? '';
			$users = $this->User->getAllUsersNotification();
			$video_details = $this->Video->getDetails($video_id);
			if (!empty($users) && !empty($video_details)) {
				foreach ($users as $user) {
					$device_token = $user['User']['device_token'] ?? '';
					if (strlen($device_token) > 15) {
						$notification = [
							'to' => $device_token,
							'notification' => [
								'title' => $title,
								'body' => $txt,
								'badge' => '1',
								'sound' => 'default',
								'icon' => '',
								'type' => 'video_posted',
								'video_id' => $video_id
							],
							'data' => [
								'title' => $title,
								'body' => $txt,
								'icon' => '',
								'badge' => '1',
								'sound' => 'default',
								'type' => 'video_posted',
								'video_id' => $video_id
							]
						];
						$this->firebase->sendPushNotification($notification);
					}
				}
			}
			$output = ['code' => 200, 'msg' => 'successfully sent'];
			$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
	}

	public function sendUserAccountNotificationToAllUsers(): void
	{
		$this->loadModel('User');
		$this->loadModel('Video');
		if ($this->request->is('post')) {
			$data = $this->request->input('json_decode', true);
			$txt = $data['text'] ?? '';
			$title = $data['title'] ?? '';
			$user_id = $data['user_id'] ?? null;
			$users = $this->User->getAllUsersNotification();
			$userDetails = $this->User->getUserDetailsFromID($user_id);
			if (!empty($users) && !empty($userDetails)) {
				foreach ($users as $user) {
					$device_token = $user['User']['device_token'] ?? '';
					if (strlen($device_token) > 15) {
						$notification = [
							'to' => $device_token,
							'notification' => [
								'title' => $title,
								'body' => $txt,
								'badge' => '1',
								'sound' => 'default',
								'icon' => '',
								'type' => 'follow',
								'user_id' => $userDetails['User']['id']
							],
							'data' => [
								'title' => $title,
								'body' => $txt,
								'icon' => '',
								'badge' => '1',
								'sound' => 'default',
								'type' => 'follow',
								'user_id' => $user_id
							]
						];
						$this->firebase->sendPushNotification($notification);
					}
				}
			}
			$output = ['code' => 200, 'msg' => 'successfully sent'];
			$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
	}

	public function promoteVideo(): void
	{
		$this->loadModel('Video');
		if ($this->request->is('post')) {
			$data = $this->request->input('json_decode', true);
			$video_id = $data['video_id'] ?? null;
			$promote = $data['promote'] ?? null;
			$details = $this->Video->getDetails($video_id);
			if (!empty($details)) {
				$videoEntity = $this->Video->get($details['Video']['id']);
				$videoEntity = $this->Video->patchEntity($videoEntity, ['promote' => $promote]);
				$this->Video->save($videoEntity);
				$updatedDetails = $this->Video->getDetails($video_id);
				$output = ['code' => 200, 'msg' => $updatedDetails];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			}
			Message::EMPTYDATA();
		}
	}

	public function addCategory(): void
	{
		if (APP_STATUS === 'demo') {
			$output = ['code' => 201, 'msg' => 'Sorry this feature is disabled in demo mode'];
			$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
			return;
		}
		$this->loadModel('Category');
		if ($this->request->is('post')) {
			$created = date('Y-m-d H:i:s');
			$data = $this->request->getData();
			$post_data['title'] = $data['title'] ?? null;
			$post_data['parent_id'] = $data['parent_id'] ?? null;
			$id = $data['id'] ?? 0;
			if ($id > 0) {
				$category = $this->Category->get($id);
				$image_db = $category->image;
				if (!empty($_FILES['file'])) {
					if (strpos($image_db, 'http') !== false) {
						Utility::deleteObjectS3($image_db);
					} elseif (strlen($image_db) > 5) {
						@unlink($image_db);
					}
					$image_details = $this->aws->fileUploadToS3Multipart('file');
					if ($image_details['code'] === 200) {
						$post_data['image'] = CLOUDFRONT_URL . '/' . $image_details['msg'];
					}
				}
				$category = $this->Category->patchEntity($category, $post_data);
				$this->Category->save($category);
				$details = $this->Category->getDetails($category->id);
				$output = ['code' => 200, 'msg' => $details];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			}
			if (!empty($_FILES['file'])) {
				if (MEDIA_STORAGE === 's3') {
					$result = $this->aws->fileUploadToS3Multipart('file');
					if ($result['code'] === 200) {
						$msg = $result['msg'];
						if (strpos($msg, 'tictic-video') !== false) {
							$key = str_replace('tictic-video', '', $msg);
							$post_data['image'] = CLOUDFRONT_URL . $key;
						} else {
							$post_data['image'] = CLOUDFRONT_URL . '/' . $msg;
						}
					}
				} else {
					$post_data['image'] = Utility::uploadAMultipartFileIntoFolder('file');
				}
			}
			$post_data['created'] = $created;
			$category = $this->Category->newEmptyEntity();
			$category = $this->Category->patchEntity($category, $post_data);
			$this->Category->save($category);
			$details = $this->Category->getDetails($category->id);
			$output = ['code' => 200, 'msg' => $details];
			$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
	}
    
	public function addCoinWorth(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('CoinWorth');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$data = $this->request->input('json_decode', true);
		$price = $data['price'] ?? null;
		$existing = $this->CoinWorth->getAll();
		if (count($existing) > 0) {
			$entity = $this->CoinWorth->get($existing['CoinWorth']['id']);
			$entity = $this->CoinWorth->patchEntity($entity, ['price' => $price]);
			$this->CoinWorth->save($entity);
		} else {
			$entity = $this->CoinWorth->newEntity(['price' => $price]);
			$this->CoinWorth->save($entity);
		}
		$details = $this->CoinWorth->getDetails($entity->id);
		$output = ['code' => 200, 'msg' => $details];
		return $this->response->withType('application/json')->withStringBody(json_encode($output));
	}

	public function showCoinWorth(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('CoinWorth');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$details = $this->CoinWorth->getAll();
		if (count($details) > 0) {
			$output = ['code' => 200, 'msg' => $details];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
		Message::EMPTYDATA();
		return $this->response;
	}

	public function showStoreSuggestion(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('StoreSuggestion');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$details = $this->StoreSuggestion->getAll();
		if (count($details) > 0) {
			$output = ['code' => 200, 'msg' => $details];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
		Message::EMPTYDATA();
		return $this->response;
	}

	public function featuredGift(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('Gift');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$data = $this->request->input('json_decode', true);
		$featured = $data['featured'] ?? null;
		$id = $data['id'] ?? null;
		$gift = $this->Gift->getDetails($id);
		if (count($gift)) {
			$entity = $this->Gift->get($id);
			$entity = $this->Gift->patchEntity($entity, ['featured' => $featured]);
			$this->Gift->save($entity);
			$gift = $this->Gift->getDetails($id);
			$output = ['code' => 200, 'msg' => $gift];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
		Message::EMPTYDATA();
		return $this->response;
	}

	public function addGift(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('Gift');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$data = $this->request->getData();
		$title = $data['title'] ?? null;
		$coin = $data['coin'] ?? null;
		$time = $data['time'] ?? null;
		$id = $data['id'] ?? null;
		$giftData = ['title' => $title, 'coin' => $coin, 'time' => $time, 'created' => date('Y-m-d H:i:s')];
		if ($id) {
			$entity = $this->Gift->get($id);
			$entity = $this->Gift->patchEntity($entity, $giftData);
			$this->Gift->save($entity);
			$details = $this->Gift->getDetails($id);
			$output = ['code' => 200, 'msg' => $details];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
		$giftData['type'] = 'image';
		$exists = $this->Gift->ifExist($title);
		if (count($exists) < 1) {
			$fileImageUrl = '';
			$fileIconUrl = '';
			$fileSoundUrl = '';
			if (MEDIA_STORAGE === 's3') {
				if ($this->request->getData('image')?->getError() === UPLOAD_ERR_OK) {
					$upload = $this->aws->fileUploadToS3Multipart('image');
					if ($upload['code'] === 200) {
						$fileKey = $upload['msg'];
						$cloudfront = str_replace('tictic-video', '', CLOUDFRONT_URL);
						$fileImageUrl = $cloudfront . '/' . $fileKey;
						if (pathinfo($fileImageUrl, PATHINFO_EXTENSION) === 'gif') {
							$giftData['type'] = 'gif';
						}
					}
				}
				if ($this->request->getData('icon')?->getError() === UPLOAD_ERR_OK) {
					$upload = $this->aws->fileUploadToS3Multipart('icon');
					if ($upload['code'] === 200) {
						$fileKey = $upload['msg'];
						$cloudfront = str_replace('tictic-video', '', CLOUDFRONT_URL);
						$fileIconUrl = $cloudfront . '/' . $fileKey;
					}
				}
				if ($this->request->getData('sound')?->getError() === UPLOAD_ERR_OK) {
					$upload = $this->aws->fileUploadToS3Multipart('sound', 'mp3');
					if ($upload['code'] === 200) {
						$fileKey = $upload['msg'];
						$cloudfront = str_replace('tictic-video', '', CLOUDFRONT_URL);
						$fileSoundUrl = $cloudfront . $fileKey;
					}
				}
			} else {
				if ($this->request->getData('image')?->getError() === UPLOAD_ERR_OK) {
					$fileImageUrl = Utility::uploadAMultipartFileIntoFolder('image', null, UPLOADS_FOLDER_URI);
					if (pathinfo($fileImageUrl, PATHINFO_EXTENSION) === 'gif') {
						$giftData['type'] = 'gif';
					}
				}
				if ($this->request->getData('icon')?->getError() === UPLOAD_ERR_OK) {
					$fileIconUrl = Utility::uploadAMultipartFileIntoFolder('icon', null, UPLOADS_FOLDER_URI);
				}
				if ($this->request->getData('sound')?->getError() === UPLOAD_ERR_OK) {
					$fileSoundUrl = Utility::uploadAMultipartFileIntoFolder('sound', null, UPLOADS_FOLDER_URI);
				}
			}
			$giftData['image'] = $fileImageUrl;
			$giftData['icon'] = $fileIconUrl;
			$giftData['sound'] = $fileSoundUrl;
			$entity = $this->Gift->newEntity($giftData);
			$this->Gift->save($entity);
			$details = $this->Gift->getDetails($entity->id);
			$output = ['code' => 200, 'msg' => $details];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
		$output = Message::DUPLICATEDATE();
		return $this->response->withType('application/json')->withStringBody($output);
	}

	public function deleteGift(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('Gift');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$data = $this->request->input('json_decode', true);
		$id = $data['id'] ?? null;
		$exists = $this->Gift->getDetails($id);
		if (count($exists) > 0) {
			$entity = $this->Gift->get($id);
			$this->Gift->delete($entity);
			$output = ['code' => 200, 'msg' => 'deleted'];
		} else {
			$output = ['code' => 201, 'msg' => 'Invalid id: Do not exist'];
		}
		return $this->response->withType('application/json')->withStringBody(json_encode($output));
	}

	public function showGifts(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('Gift');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$data = $this->request->input('json_decode', true);
		$gifts = isset($data['id']) ? $this->Gift->getDetails($data['id']) : $this->Gift->getAll();
		$output = ['code' => 200, 'msg' => $gifts];
		return $this->response->withType('application/json')->withStringBody(json_encode($output));
	}

	public function showWithdrawRequest(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('WithdrawRequest');
		if (!$this->request->is('post')) {
			throw new \Cake\Http\Exception\MethodNotAllowedException();
		}
		$data = $this->request->input('json_decode', true);
		if (isset($data['user_id'])) {
			$requests = $this->WithdrawRequest->getUserPendingWithdrawRequest($data['user_id']);
		} elseif (isset($data['id'])) {
			$requests = $this->WithdrawRequest->getDetails($data['id']);
		} else {
			$requests = $this->WithdrawRequest->getAllPendingRequests(0);
		}
		$output = ['code' => 200, 'msg' => $requests];
		return $this->response->withType('application/json')->withStringBody(json_encode($output));
	}
    
		public function showSettings(): \Cake\Http\Response
		{
			$this->request->allowMethod(['post']);
			$this->autoRender = false;
			$data = $this->request->input('json_decode', true);
			$details = !empty($data['id']) ? $this->Setting->getDetails($data['id']) : $this->Setting->getAll();
			if (!empty($details)) {
				$output = ['code' => 200, 'msg' => $details];
			} else {
				$output = Message::EMPTYDATA();
			}
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}

		public function deleteSetting(): \Cake\Http\Response
		{
			$this->request->allowMethod(['post']);
			$this->autoRender = false;
			$data = $this->request->input('json_decode', true);
			$details = $this->Setting->getDetails($data['id'] ?? null);
			if (!empty($details)) {
				$entity = $this->Setting->get($data['id']);
				$this->Setting->delete($entity);
				$output = ['code' => 200, 'msg' => 'deleted'];
			} else {
				$output = ['code' => 201, 'msg' => 'Invalid id: Do not exist'];
			}
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}

		public function addSetting(): \Cake\Http\Response
		{
			$this->request->allowMethod(['post']);
			$this->autoRender = false;
			$data = $this->request->input('json_decode', true);
			$report = [
				'type' => $data['type'],
				'value' => $data['value'],
				'created' => date('Y-m-d H:i:s')
			];
			if (!empty($data['id'])) {
				$entity = $this->Setting->get($data['id']);
				$entity = $this->Setting->patchEntity($entity, $report);
				$this->Setting->save($entity);
				$details = $this->Setting->getDetails($data['id']);
				$output = ['code' => 200, 'msg' => $details];
			} else {
				$duplicate = $this->Setting->checkDuplicate($data['type']);
				if (empty($duplicate)) {
					$entity = $this->Setting->newEntity($report);
					$this->Setting->save($entity);
					$details = $this->Setting->getDetails($entity->id);
					$output = ['code' => 200, 'msg' => $details];
				} else {
					$output = Message::DUPLICATEDATE();
				}
			}
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}

		public function withdrawRequestApproval(): \Cake\Http\Response
		{
			$this->request->allowMethod(['post']);
			$this->autoRender = false;
			$data = $this->request->input('json_decode', true);
			$withdraw = $this->WithdrawRequest->get($data['id']);
			if ($withdraw) {
				if ($data['status'] == 1) {
					$user = $this->User->get($withdraw->user_id);
					$user = $this->User->patchEntity($user, ['wallet' => 0, 'reset_wallet_datetime' => date('Y-m-d H:i:s')]);
					$this->User->save($user);
				}
				$withdraw = $this->WithdrawRequest->patchEntity($withdraw, ['status' => $data['status'], 'updated' => date('Y-m-d H:i:s')]);
				$this->WithdrawRequest->save($withdraw);
				$details = $this->WithdrawRequest->getDetails($data['id']);
				$output = ['code' => 200, 'msg' => $details];
			} else {
				$output = Message::EMPTYDATA();
			}
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}

		public function showSounds(): \Cake\Http\Response
		{
			$this->request->allowMethod(['post']);
			$this->autoRender = false;
			$data = $this->request->input('json_decode', true);
			if (isset($data['publish'])) {
				$sounds = $this->Sound->getSoundsAccordingToStatus($data['publish']);
			} else {
				$sounds = $this->Sound->getSounds();
			}
			$output = ['code' => 200, 'msg' => $sounds];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}

		public function showSoundSections(): \Cake\Http\Response
		{
			$this->request->allowMethod(['post']);
			$this->autoRender = false;
			$data = $this->request->input('json_decode', true);
			if (isset($data['id'])) {
				$sounds = $this->SoundSection->getDetails($data['id']);
			} else {
				$sounds = $this->SoundSection->getAll();
				if (count($sounds) > 1) {
					foreach ($sounds as $key => $section) {
						$sounds[$key]['SoundSection']['total_sounds'] = $this->Sound->getSoundsCount($section['SoundSection']['id']);
					}
				} elseif (count($sounds) === 1) {
					$sounds[0]['SoundSection']['total_sounds'] = $this->Sound->getSoundsCount($sounds[0]['SoundSection']['id']);
				} else {
					$output = Message::EMPTYDATA();
					return $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}
			$output = ['code' => 200, 'msg' => $sounds];
			return $this->response->withType('application/json')->withStringBody(json_encode($output));
		}
    
		public function deleteCategory(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Categories');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['id'] ?? null;
				$code = 400;
				$msg = 'Invalid id';
				if ($id) {
					try {
						$entity = $this->Categories->get($id);
						if ($this->Categories->delete($entity)) {
							$code = 200;
							$msg = 'deleted';
						} else {
							$code = 500;
							$msg = 'Could not delete';
						}
					} catch (\Exception $e) {
						$code = 404;
						$msg = 'Not found';
					}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function deletePromotion(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Promotions');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['id'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($id) {
					try {
						$entity = $this->Promotions->get($id);
						if ($this->Promotions->delete($entity)) {
							$code = 200;
							$msg = 'deleted';
						}
					} catch (\Exception $e) {}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function showAllPromotions(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Promotions');
			$this->loadModel('VideoWatches');
			$this->loadModel('VideoLikes');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$promotions = $this->Promotions->getAll();
				$total_coins_spent = 0;
				$total_video_views = 0;
				$total_destination_tap = 0;
				$total_video_likes = 0;
				if (!empty($promotions)) {
					foreach ($promotions as $key => $val) {
						$videoId = $val['Promotion']['video_id'];
						$start = $val['Promotion']['start_datetime'];
						$end = $val['Promotion']['end_datetime'];
						$coin = $val['Promotion']['coin'];
						$destinationTap = $val['Promotion']['destination_tap'];
						$views = $this->VideoWatches->countWatchVideos([$videoId], $start, $end);
						$likes = $this->VideoLikes->countLikesBetweenDatetime([$videoId], $start, $end);
						$promotions[$key]['Promotion']['views'] = $views;
						$total_coins_spent += $coin;
						$total_video_views += $views;
						$total_destination_tap += $destinationTap;
						$total_video_likes += $likes;
					}
				}
				$msg = [
					'Promotions' => $promotions,
					'Stats' => [
						'coins_spent' => $total_coins_spent,
						'total_video_views' => $total_video_views,
						'total_destination_tap' => $total_destination_tap,
						'total_video_likes' => $total_video_likes
					]
				];
				$payload = ['code' => 200, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function approveBusiness(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Businesses');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['business_id'] ?? null;
				$statusValue = $data['status'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($id) {
					try {
						$entity = $this->Businesses->get($id);
						$entity->status = $statusValue;
						if ($this->Businesses->save($entity)) {
							$details = $this->Businesses->get($id)->toArray();
							$code = 200;
							$msg = $details;
						}
					} catch (\Exception $e) {}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function approvePromotion(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Promotions');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['promotion_id'] ?? null;
				$active = $data['active'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($id) {
					try {
						$entity = $this->Promotions->get($id);
						$entity->active = $active;
						if ($this->Promotions->save($entity)) {
							$details = $this->Promotions->get($id)->toArray();
							$code = 200;
							$msg = $details;
						}
					} catch (\Exception $e) {}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function publishSound(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Sounds');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['sound_id'] ?? null;
				$publish = $data['publish'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($id) {
					try {
						$entity = $this->Sounds->get($id);
						$entity->publish = $publish;
						if ($this->Sounds->save($entity)) {
							$details = $this->Sounds->get($id)->toArray();
							$code = 200;
							$msg = $details;
						}
					} catch (\Exception $e) {}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function addSoundInSection(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Sounds');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['sound_id'] ?? null;
				$sectionId = $data['sound_section_id'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($id) {
					try {
						$entity = $this->Sounds->get($id);
						$entity->sound_section_id = $sectionId;
						if ($this->Sounds->save($entity)) {
							$details = $this->Sounds->get($id)->toArray();
							$code = 200;
							$msg = $details;
						}
					} catch (\Exception $e) {}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function deleteVideo(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Videos');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$videoId = $data['video_id'] ?? null;
				$code = 400;
				$msg = 'Invalid id';
				if ($videoId) {
					if (APP_STATUS === 'demo') {
						$code = 201;
						$msg = 'You cannot delete account in demo account. You need to contact qboxus for that';
					} else {
						try {
							$details = $this->Videos->getDetails($videoId);
							if (!empty($details)) {
								$videoUrl = $details['Video']['video'];
								$thumbUrl = $details['Video']['thum'];
								$gifUrl = $details['Video']['gif'];
								if (strpos($videoUrl, 'http') !== false) {
									$r1 = $this->aws->deleteObjectS3($videoUrl);
									$r2 = $this->aws->deleteObjectS3($thumbUrl);
									$r3 = $this->aws->deleteObjectS3($gifUrl);
									if ($r1 && $r2 && $r3) {
										$code = 200;
										$msg = 'deleted successfully';
									} else {
										$code = 201;
										$msg = 'something went wrong in deleting the file from the cdn';
									}
								} else {
									unlink($videoUrl);
									unlink($thumbUrl);
									unlink($gifUrl);
									$code = 200;
									$msg = 'successfully deleted';
								}
								$entity = $this->Videos->get($videoId);
								$this->Videos->delete($entity);
								Cache::delete('video_details_' . $videoId);
							} else {
								$code = 200;
								$msg = 'video has been already deleted';
							}
						} catch (\Exception $e) {
							$code = 500;
							$msg = 'Error processing request';
						}
					}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function showVideoComments(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('VideoComments');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$videoId = $data['video_id'] ?? null;
				$comments = $videoId ? $this->VideoComments->getVideoComments($videoId) : [];
				$payload = ['code' => 200, 'msg' => $comments];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function deleteSound(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('Sounds');
			$this->loadModel('Videos');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$soundId = $data['sound_id'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($soundId) {
					try {
						$details = $this->Sounds->getDetails($soundId);
						if (!empty($details)) {
							$audioUrl = $details['Sound']['audio'];
							if (strpos($audioUrl, 'http') !== false) {
								if ($this->aws->deleteObjectS3($audioUrl)) {
									$code = 200;
									$msg = 'deleted successfully';
								} else {
									$code = 201;
									$msg = 'something went wrong in deleting the file from the cdn';
								}
							} else {
								Utility::unlinkFile($audioUrl);
								$code = 200;
								$msg = 'successfully deleted';
							}
							$entity = $this->Sounds->get($soundId);
							$this->Sounds->delete($entity);
							$allVideos = $this->Videos->getAllVideosAgainstSoundID($soundId);
							if (!empty($allVideos)) {
								$videoIds = [];
								foreach ($allVideos as $val) {
									$videoIds[] = $val['Video']['id'];
								}
								$this->Videos->updateSoundIDs($videoIds);
							}
						}
					} catch (\Exception $e) {
						$code = 500;
						$msg = 'Error processing request';
					}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function showAppSlider(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('AppSliders');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$images = $this->AppSliders->getAll();
				$payload = ['code' => 200, 'msg' => $images];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}

		public function deleteAppSlider(): \Psr\Http\Message\ResponseInterface
		{
			$this->loadModel('AppSliders');
			$request = $this->getRequest();
			if ($request->is('post')) {
				$data = $request->getData();
				$id = $data['id'] ?? null;
				$code = 201;
				$msg = 'Invalid id: Do not exist';
				if ($id) {
					try {
						$entity = $this->AppSliders->get($id);
						if ($this->AppSliders->delete($entity)) {
							$code = 200;
							$msg = 'deleted';
						}
					} catch (\Exception $e) {}
				}
				$payload = ['code' => $code, 'msg' => $msg];
				return $this->getResponse()
							->withType('application/json')
							->withStringBody(json_encode($payload));
			}
			return $this->getResponse();
		}
    
				public function addAudience(): void
				{
					$this->loadModel('Audience');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$audience = $this->Audience->newEntity([
							'name' => $data['name'],
							'user_id' => $data['user_id'],
							'min_age' => $data['min_age'],
							'max_age' => $data['max_age'],
							'gender' => strtolower($data['gender']),
							'created' => date('Y-m-d H:i:s')
						]);
						if ($this->Audience->save($audience)) {
							$details = $this->Audience->getDetails($audience->id);
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							echo json_encode(['code' => 400, 'msg' => 'Unable to save audience']);
						}
						return;
					}
				}

				public function addDefaultAudience(): int
				{
					$this->loadModel('Audience');
					$audience = $this->Audience->newEntity([
						'name' => 'default',
						'user_id' => 0,
						'min_age' => 18,
						'max_age' => 100,
						'gender' => 'all',
						'created' => date('Y-m-d H:i:s')
					]);
					$this->Audience->save($audience);
					return $audience->id;
				}

				public function deleteAudience(): void
				{
					$this->loadModel('Audience');
					$this->loadModel('Promotion');
					if ($this->request->is('post')) {
						$id = $this->request->getData('id');
						$created = date('Y-m-d H:i:s');
						$details = $this->Promotion->getActivePromotionAudience($id, $created);
						if (count($details) < 1) {
							$audience = $this->Audience->get($id);
							if ($this->Audience->delete($audience)) {
								echo json_encode(['code' => 200, 'msg' => 'deleted']);
							} else {
								echo json_encode(['code' => 400, 'msg' => 'Unable to delete audience']);
							}
						} else {
							echo json_encode(['code' => 201, 'msg' => 'You cannot delete an audience since your ad is active against this audience.']);
						}
						return;
					}
				}

				public function showAudiences(): void
				{
					$this->loadModel('Audience');
					if ($this->request->is('post')) {
						$user_id = $this->request->getData('user_id');
						$audiences = $this->Audience->getUserAudiences($user_id);
						echo json_encode(['code' => 200, 'msg' => $audiences]);
						return;
					}
				}

				public function addPromotion(): void
				{
					$this->loadModel('Promotion');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$promotion_data = [];
						if ($data['destination'] === 'website') {
							$promotion_data['website_url'] = $data['website_url'];
							$promotion_data['action_button'] = $data['action_button'];
						}
						$audience_id = $data['audience_id'] < 1 ? $this->addDefaultAudience() : $data['audience_id'];
						$promotion_data += [
							'video_id' => $data['video_id'],
							'user_id' => $data['user_id'],
							'destination' => $data['destination'],
							'coin' => $data['coin'],
							'audience_id' => $audience_id,
							'total_reach' => $data['total_reach'],
							'start_datetime' => $data['start_datetime'],
							'end_datetime' => $data['end_datetime'],
							'active' => 0,
							'created' => date('Y-m-d H:i:s')
						];
						$promotion = $this->Promotion->newEntity($promotion_data);
						if ($this->Promotion->save($promotion)) {
							$details = $this->Promotion->getDetails($promotion->id);
							$wallet_total = $this->walletTotal($data['user_id']);
							$details['User']['wallet'] = $wallet_total['total'];
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							echo json_encode(['code' => 400, 'msg' => 'Unable to save promotion']);
						}
						return;
					}
				}

				public function addAppSlider(): void
				{
					$this->loadModel('AppSlider');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$user_id = 1;
						$image_data = ['url' => $data['url']];
						if (!empty($data['id'])) {
							$existing = $this->AppSlider->get($data['id']);
							if (!empty($data['image'])) {
								@unlink($existing->image);
								$filePath = Utility::uploadFileintoFolder($user_id, $data['image'], UPLOADS_FOLDER_URI);
								$image_data['image'] = $filePath;
							}
							$slider = $this->AppSlider->patchEntity($existing, $image_data);
						} else {
							if (!empty($data['image'])) {
								$filePath = Utility::uploadFileintoFolder($user_id, $data['image'], UPLOADS_FOLDER_URI);
								$image_data['image'] = $filePath;
							}
							$slider = $this->AppSlider->newEntity($image_data);
						}
						if ($this->AppSlider->save($slider)) {
							$app_slider = $this->AppSlider->get($slider->id);
							echo json_encode(['code' => 200, 'msg' => $app_slider]);
						} else {
							echo json_encode(['code' => 400, 'msg' => 'Unable to save slider']);
						}
						return;
					}
				}

				public function addAppSlider2(): void
				{
					$this->loadModel('AppSlider');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						if (MEDIA_STORAGE === 's3') {
							$result = $this->aws->fileUploadToS3Multipart('file', 'png');
							$file_url = $result['code'] === 200
								? (strpos($result['msg'], 'tictic-video') !== false
									? CLOUDFRONT_URL . str_replace('tictic-video', '', $result['msg'])
									: CLOUDFRONT_URL . '/' . $result['msg'])
								: null;
						} else {
							$file_url = Utility::uploadAMultipartFileIntoFolder('file', null, UPLOADS_FOLDER_URI);
						}
						$slider = $this->AppSlider->newEntity(['image' => $file_url, 'url' => $data['url']]);
						if ($this->AppSlider->save($slider)) {
							$app_slider = $this->AppSlider->get($slider->id);
							echo json_encode(['code' => 200, 'msg' => $app_slider]);
						} else {
							echo json_encode(['code' => 400, 'msg' => 'Unable to save slider']);
						}
						return;
					}
				}

				public function showCategories(): void
				{
					$this->loadModel('Category');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$details = !empty($data['id'])
							? $this->Category->get($data['id'])
							: $this->Category->getAll($data['parent_id']);
						if (!empty($details)) {
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							Message::EMPTYDATA();
						}
						return;
					}
				}

				public function deleteSticker(): void
				{
					$this->loadModel('Sticker');
					if ($this->request->is('post')) {
						$id = $this->request->getData('id');
						try {
							$sticker = $this->Sticker->get($id);
							if ($this->Sticker->delete($sticker)) {
								echo json_encode(['code' => 200, 'msg' => 'deleted']);
							} else {
								echo json_encode(['code' => 400, 'msg' => 'Unable to delete sticker']);
							}
						} catch (\Cake\Datasource\Exception\RecordNotFoundException $e) {
							echo json_encode(['code' => 201, 'msg' => 'Invalid id: Do not exist']);
						}
						return;
					}
				}

				public function deleteTopic(): void
				{
					$this->loadModel('Topic');
					if ($this->request->is('post')) {
						$id = $this->request->getData('id');
						try {
							$topic = $this->Topic->get($id);
							if ($this->Topic->delete($topic)) {
								echo json_encode(['code' => 200, 'msg' => 'deleted']);
							} else {
								echo json_encode(['code' => 400, 'msg' => 'Unable to delete topic']);
							}
						} catch (\Cake\Datasource\Exception\RecordNotFoundException $e) {
							echo json_encode(['code' => 201, 'msg' => 'Invalid id: Do not exist']);
						}
						return;
					}
				}

				public function addTopic(): void
				{
					$this->loadModel('Topic');
					if ($this->request->is('post')) {
						$title = $this->request->getData('title');
						$existing = $this->Topic->checkDuplicate($title);
						if ($existing) {
							$topic = $this->Topic->patchEntity(
								$this->Topic->get($existing['Topic']['id']),
								['title' => $title]
							);
						} else {
							$topic = $this->Topic->newEntity(['title' => $title]);
						}
						if ($this->Topic->save($topic)) {
							$details = $this->Topic->get($topic->id);
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							echo json_encode(['code' => 400, 'msg' => 'Unable to save topic']);
						}
						return;
					}
				}

				public function showTopics(): void
				{
					$this->loadModel('Topic');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$details = !empty($data['id'])
							? $this->Topic->get($data['id'])
							: $this->Topic->getAll();
						if (!empty($details)) {
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							Message::EMPTYDATA();
						}
						return;
					}
				}

				public function showStickers(): void
				{
					$this->loadModel('Sticker');
					if ($this->request->is('post')) {
						$images = $this->Sticker->getAllAdmin();
						echo json_encode(['code' => 200, 'msg' => $images]);
						return;
					}
				}

				public function addSticker(): void
				{
					$this->loadModel('Sticker');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$folder = UPLOADS_FOLDER_URI;
						if (!empty($data['id'])) {
							$sticker = $this->Sticker->get($data['id']);
							if (!empty($sticker->image)) {
								if (strpos($sticker->image, 'http') !== false) {
									$this->aws->deleteObjectS3($sticker->image);
								} else {
									@unlink($sticker->image);
								}
							}
							$filePath = Utility::uploadFileintoFolderDir($data['image'], $folder, 'png');
							$stickerData = ['image' => $filePath, 'title' => $data['title']];
							if (MEDIA_STORAGE === 's3') {
								$res = $this->aws->fileUploadToS3($filePath, 'png');
								if ($res['code'] === 200) {
									$stickerData['image'] = Utility::getCloudFrontUrl($res['msg'], '/sticker');
								}
								@unlink($filePath);
							}
							$sticker = $this->Sticker->patchEntity($sticker, $stickerData);
						} else {
							$filePath = Utility::uploadFileintoFolderDir($data['image'], $folder, 'png');
							if (MEDIA_STORAGE === 's3') {
								$res = $this->aws->fileUploadToS3($filePath, 'png');
								$imagePath = $res['code'] === 200
									? Utility::getCloudFrontUrl($res['msg'], '/sticker')
									: null;
								@unlink($filePath);
							} else {
								$imagePath = $filePath;
							}
							$sticker = $this->Sticker->newEntity(['image' => $imagePath, 'title' => $data['title']]);
						}
						if ($this->Sticker->save($sticker)) {
							$saved = $this->Sticker->get($sticker->id);
							echo json_encode(['code' => 200, 'msg' => $saved]);
						} else {
							echo json_encode(['code' => 400, 'msg' => 'Unable to save sticker']);
						}
						return;
					}
				}

				public function blockVideo(): void
				{
					$this->loadModel('Video');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$video = $this->Video->get($data['video_id']);
						$video = $this->Video->patchEntity($video, ['block' => $data['block']]);
						if ($this->Video->save($video)) {
							$details = $this->Video->get($data['video_id']);
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							Message::EMPTYDATA();
						}
						return;
					}
				}

				public function test1(): void
				{
					$data = Utility::get_hashtags('hello Hi I am #irfan #but ');
					pr($data);
				}

				public function editVideo(): void
				{
					$this->loadModel('Video');
					$this->loadModel('Sound');
					$this->loadModel('Hashtag');
					$this->loadModel('HashtagVideo');
					$this->loadModel('User');
					$this->loadModel('Notification');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$video = $this->Video->get($data['video_id']);
						$video = $this->Video->patchEntity($video, [
							'description' => $data['description'],
							'privacy_type' => $data['privacy_type'],
							'allow_comments' => $data['allow_comments'],
							'allow_duet' => $data['allow_duet'],
							'user_id' => $data['user_id'],
							'views' => $data['views'],
							'created' => date('Y-m-d H:i:s')
						]);
						if ($this->Video->save($video)) {
							$details = $this->Video->get($data['video_id']);
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							echo Message::DATASAVEERROR();
						}
						return;
					}
				}

				public function editVideoViews(): void
				{
					$this->loadModel('Video');
					if ($this->request->is('post')) {
						$data = $this->request->getData();
						$video = $this->Video->get($data['video_id']);
						$video = $this->Video->patchEntity($video, ['view' => $data['view']]);
						if ($this->Video->save($video)) {
							$details = $this->Video->get($data['video_id']);
							echo json_encode(['code' => 200, 'msg' => $details]);
						} else {
							Message::EMPTYDATA();
						}
						return;
					}
				}
    
					public function addSound(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('Sound');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data = $request->getData();
							$ffmpeg = new Ffmpeg();
							$id = $data['id'] ?? 0;
							$soundData = [
								'name'             => $data['name'],
								'description'      => $data['description'],
								'uploaded_by'      => 'admin',
								'publish'          => 1,
								'created'          => date('Y-m-d H:i:s'),
								'sound_section_id' => $data['sound_section_id']
							];
							if ($id > 0) {
								$soundEntity = $this->Sound->get($id);
								$soundEntity = $this->Sound->patchEntity($soundEntity, $soundData);
								$this->Sound->save($soundEntity);
								$details = $this->Sound->getDetails($id);
								$output = ['code' => 200, 'msg' => $details];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							}
							if (MEDIA_STORAGE === 's3') {
								$soundFile = $this->Aws->fileUploadToS3Multipart($request->getData('audio'));
								$thumbFile = $this->Aws->fileUploadToS3Multipart($request->getData('thum'));
								if ($soundFile['code'] === 200 && $thumbFile['code'] === 200) {
									$soundKey   = $soundFile['msg'];
									$thumbKey   = $thumbFile['msg'];
									$cloudfront = str_replace('tictic-video', '', CLOUDFRONT_URL);
									$fileSound  = $cloudfront . '/' . $soundKey;
									$fileThumb  = $cloudfront . '/' . $thumbKey;
									$filename   = basename($soundKey);
									$r = $this->Aws->transcodeAudio($filename);
									if ($r['code'] < 201) {
										$soundData['job_id'] = $r['msg'];
									}
									if ($r['code'] === 200) {
										$soundData['audio']    = $fileSound;
										$soundData['thum']     = $fileThumb;
										$soundData['duration'] = $ffmpeg->getDurationofAudioFile($fileSound);
									} else {
										$output = ['code' => 201, 'msg' => $r['msg']];
										return $this->response->withType('application/json')->withStringBody(json_encode($output));
									}
								}
							} else {
								$audioPath = Utility::uploadAMultipartFileIntoFolder($request->getData('audio'), null, UPLOADS_FOLDER_URI);
								$thumbPath = Utility::uploadAMultipartFileIntoFolder($request->getData('thum'),  null, UPLOADS_FOLDER_URI);
								$soundData['audio']    = $audioPath;
								$soundData['thum']     = $thumbPath;
								$soundData['duration'] = $ffmpeg->getDurationofAudioFile($audioPath);
							}
							$soundEntity = $this->Sound->newEmptyEntity();
							$soundEntity = $this->Sound->patchEntity($soundEntity, $soundData);
							$this->Sound->save($soundEntity);
							$newId   = $soundEntity->id;
							$details = $this->Sound->getDetails($newId);
							$output  = ['code' => 200, 'msg' => $details];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
						return $this->response;
					}

					public function showOrders(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('Order');
						$this->loadModel('RiderOrder');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data   = $request->getData();
							$orders = [];
							if (!empty($data['status'])) {
								$orders = $this->Order->getOrdersAccordingToStatus($data['status']);
							}
							if (!empty($data['order_id'])) {
								$orders = $this->Order->getDetails($data['order_id']);
								if (!empty($orders)) {
									$rider = $this->RiderOrder->getRiderOrderAgainstOrderID($orders['Order']['id']);
									if (!empty($rider)) {
										$orders['Order']['RiderOrder'] = $rider['RiderOrder'];
									}
								}
							}
							if (!empty($orders)) {
								$output = ['code' => 200, 'msg' => $orders];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} else {
								Message::EMPTYDATA();
							}
						}
						return $this->response;
					}

					public function adminResponseAgainstOrder(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('Order');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data  = $request->getData();
							$order = $this->Order->get($data['order_id']);
							$order = $this->Order->patchEntity($order, ['status' => $data['status']]);
							$this->Order->save($order);
							$details = $this->Order->getDetails($data['order_id']);
							$output  = ['code' => 200, 'msg' => $details];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
						return $this->response;
					}

					public function assignOrderToRider(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('RiderOrder');
						$this->loadModel('Order');
						$this->loadModel('User');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data = $request->getData();
							if (!empty($data['id'])) {
								$this->RiderOrder->delete($data['id']);
							}
							$entity = $this->RiderOrder->newEmptyEntity();
							$entity = $this->RiderOrder->patchEntity($entity, [
								'rider_user_id'     => $data['rider_user_id'],
								'order_id'          => $data['order_id'],
								'assign_date_time'  => date('Y-m-d H:i:s')
							]);
							if ($this->RiderOrder->save($entity)) {
								$details = $this->RiderOrder->getDetails($entity->id);
								$msg     = 'You have received the new order request';
								$notification = [
									'to'           => $details['Rider']['device_token'],
									'notification' => ['title' => $msg, 'body' => '', 'badge' => '1', 'sound' => 'default', 'icon' => '', 'type' => ''],
									'data'         => ['title' => $msg, 'body' => '', 'icon' => '', 'badge' => '1', 'sound' => 'default', 'type' => '']
								];
								$this->Firebase->sendPushNotification($notification);
								$output = ['code' => 200, 'msg' => $details];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} else {
								echo Message::DUPLICATEDATE();
							}
						}
						return $this->response;
					}

					public function showRiderOrders(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('RiderOrder');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data   = $request->getData();
							$orders = $this->RiderOrder->getAllRiderOrders($data['rider_user_id']);
							$output = ['code' => 200, 'msg' => $orders];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
						return $this->response;
					}

					public function showReportReasons(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('ReportReason');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data    = $request->getData();
							$details = !empty($data['id'])
								? $this->ReportReason->getDetails($data['id'])
								: $this->ReportReason->getAll();
							if (!empty($details)) {
								$output = ['code' => 200, 'msg' => $details];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} else {
								Message::EMPTYDATA();
							}
						}
						return $this->response;
					}

					public function showReportedVideos(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('ReportVideo');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data    = $request->getData();
							$details = !empty($data['id'])
								? $this->ReportVideo->getDetails($data['id'])
								: $this->ReportVideo->getAll();
							if (!empty($details)) {
								$output = ['code' => 200, 'msg' => $details];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} else {
								Message::EMPTYDATA();
							}
						}
						return $this->response;
					}

					public function deleteReportedVideo(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('ReportVideo');
						$this->loadModel('Video');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data    = $request->getData();
							$details = $this->ReportVideo->getDetails($data['id']);
							if (!empty($details)) {
								$videoId = $details['Video']['id'];
								$this->ReportVideo->delete($data['id']);
								$this->Video->delete($videoId);
								$this->Firebase->deleteVideoFromFirestore($videoId);
								Cache::delete('video_details_' . $videoId);
								$output = ['code' => 200, 'msg' => 'deleted'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} else {
								$output = ['code' => 201, 'msg' => 'Invalid id: Do not exist'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							}
						}
						return $this->response;
					}

					public function deleteReportedUser(): \Psr\Http\Message\ResponseInterface {
						$this->loadModel('ReportUser');
						$this->loadModel('User');
						$this->loadModel('Follower');
						$this->loadModel('Store');
						$request = $this->getRequest();
						if ($request->is('post')) {
							$data    = $request->getData();
							$details = $this->ReportUser->getDetails($data['id']);
							if (!empty($details)) {
								$this->ReportUser->delete($data['id']);
								$userId        = $details['ReportUser']['report_user_id'];
								$storeDetails  = $this->Store->getObjectsForCache($userId, 'user');
								if (!empty($storeDetails)) {
									$this->Store->delete($storeDetails['Store']['id']);
									Cache::delete('store_details' . $storeDetails['Store']['id']);
								}
								$this->User->delete($userId);
								$this->Follower->deleteFollowerAgainstUserID($userId);
								$this->Firebase->deleteUser($details['Report']['auth_token']);
								Cache::delete('user_details_' . $userId);
								$output = ['code' => 200, 'msg' => 'deleted'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} else {
								$output = ['code' => 201, 'msg' => 'Invalid id: Do not exist'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							}
						}
						return $this->response;
					}
    
					public function count_(mixed $array): int {
						return is_array($array) ? count($array) : 0;
					}

					public function showReportedUsers(): void {
						$this->loadModel('ReportUser');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$details = isset($data['id']) ? $this->ReportUser->getDetails($data['id']) : $this->ReportUser->getAll();
							if (!empty($details)) {
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							Message::EMPTYDATA();
							return;
						}
					}

					public function addReportReason(): void {
						$this->loadModel('ReportReason');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$report = ['title' => $data['title'], 'created' => date('Y-m-d H:i:s')];
							$existing = $this->ReportReason->checkDuplicate($data['title']);
							if (isset($data['id'])) {
								$this->ReportReason->id = $data['id'];
								$this->ReportReason->save($report);
								$details = $this->ReportReason->getDetails($data['id']);
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							if ($existing < 1) {
								$this->ReportReason->save($report);
								$id = $this->ReportReason->id;
								$details = $this->ReportReason->getDetails($id);
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							echo Message::DUPLICATEDATE();
							return;
						}
					}

					public function editUser(): void {
						$this->loadModel('User');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$user = [
								'first_name' => $data['first_name'],
								'last_name'  => $data['last_name'],
								'gender'     => $data['gender'],
								'bio'        => $data['bio'],
								'website'    => $data['website'],
								'email'      => $data['email']
							];
							$user_id = $data['user_id'];
							$existing = $this->User->getUserDetailsFromID($user_id);
							if (!empty($existing)) {
								$this->User->id = $user_id;
								$this->User->save($user);
								$details = $this->User->getUserDetailsFromID($user_id);
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							Message::EMPTYDATA();
							return;
						}
					}

					public function blockUser(): void {
						$this->loadModel('User');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$user_id = $data['user_id'];
							$active  = $data['active'];
							$existing = $this->User->getUserDetailsFromID($user_id);
							if (!empty($existing)) {
								$this->User->id = $user_id;
								$this->User->saveField('active', $active);
								$details = $this->User->getUserDetailsFromID($user_id);
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							Message::EMPTYDATA();
							return;
						}
					}

					public function addHtmlPage(): void {
						$this->loadModel('HtmlPage');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$html = [
								'name'    => $data['name'],
								'text'    => $data['text'],
								'created' => date('Y-m-d H:i:s')
							];
							$existing = $this->HtmlPage->ifExist($data['name']);
							if (empty($existing)) {
								$this->HtmlPage->save($html);
								$id = $this->HtmlPage->id;
								$details = $this->HtmlPage->getDetails($id);
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							$this->HtmlPage->id = $existing['HtmlPage']['id'];
							$this->HtmlPage->save($html);
							$details = $this->HtmlPage->getDetails($existing['HtmlPage']['id']);
							echo json_encode(['code' => 200, 'msg' => $details]);
							return;
						}
					}

					public function showHtmlPage(): void {
						$this->loadModel('HtmlPage');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$existing = $this->HtmlPage->ifExist($data['name']);
							if (!empty($existing)) {
								echo json_encode(['code' => 200, 'msg' => $existing]);
								return;
							}
							Message::EMPTYDATA();
							return;
						}
					}

					public function addUser(): void {
						$this->loadModel('User');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							if (!empty($data['email']) && !empty($data['password'])) {
								$user = [
									'email'      => $data['email'],
									'password'   => $data['password'],
									'first_name' => $data['first_name'],
									'last_name'  => $data['last_name'],
									'role'       => $data['role'],
									'created'    => date('Y-m-d H:i:s')
								];
								if ($this->User->isEmailAlreadyExist($data['email']) > 0) {
									echo Message::DATAALREADYEXIST();
									return;
								}
								if (!$this->User->save($user)) {
									echo Message::DATASAVEERROR();
									return;
								}
								$id = $this->User->id;
								$details = $this->User->getUserDetailsFromID($id);
								echo json_encode(['code' => 200, 'msg' => $details]);
								return;
							}
							echo Message::ERROR();
							return;
						}
					}

					public function showFollowers(): void {
						$this->loadModel('Follower');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$followers = $this->Follower->getUserFollowersAdmin($data['user_id']);
							if ($this->count_($followers) < 1) {
								Message::EMPTYDATA();
								return;
							}
							echo json_encode(['code' => 200, 'msg' => $followers]);
							return;
						}
					}

					public function showFollowing(): void {
						$this->loadModel('Follower');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$following = $this->Follower->getUserFollowingAdmin($data['user_id']);
							if ($this->count_($following) < 1) {
								Message::EMPTYDATA();
								return;
							}
							echo json_encode(['code' => 200, 'msg' => $following]);
							return;
						}
					}

					public function showLikedVideos(): void {
						$this->loadModel('VideoLike');
						if ($this->getRequest()->is('post')) {
							$data = json_decode($this->getRequest()->getInput(), true);
							$videos = $this->VideoLike->getUserVideoLikes($data['user_id']);
							if ($this->count_($videos) < 1) {
								Message::EMPTYDATA();
								return;
							}
							echo json_encode(['code' => 200, 'msg' => $videos]);
							return;
						}
					}
    
						public function showFavouriteVideos(): ResponseInterface
						{
							$this->loadModel("VideoFavourite");
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$userId = $this->request->getData('user_id');
							$videosFavourite = $this->VideoFavourite->getUserFavouriteVideosAdmin($userId);
							if (empty($videosFavourite)) {
								return $this->response->withStatus(204);
							}
							$output = ['code' => 200, 'msg' => $videosFavourite];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}

						public function showUserVideos(): ResponseInterface
						{
							$this->loadModel("Video");
							$this->loadModel("VideoComment");
							$this->loadModel("VideoLike");
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$userId = $this->request->getData('user_id');
							$videosPublic = $this->Video->getUserPublicVideosAdmin($userId);
							foreach ($videosPublic as $key => $val) {
								$videosPublic[$key]['Video']['comment_count'] = $this->VideoComment->countComments($val['Video']['id']);
								$videosPublic[$key]['Video']['like_count'] = $this->VideoLike->countLikes($val['Video']['id']);
							}
							$videosPrivate = $this->Video->getUserPrivateVideosAdmin($userId);
							foreach ($videosPrivate as $key => $val) {
								$videosPrivate[$key]['Video']['comment_count'] = $this->VideoComment->countComments($val['Video']['id']);
								$videosPrivate[$key]['Video']['like_count'] = $this->VideoLike->countLikes($val['Video']['id']);
							}
							$userDetails = ['public' => $videosPublic, 'private' => $videosPrivate];
							$output = ['code' => 200, 'msg' => $userDetails];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}

						public function showUserDetail(): ResponseInterface
						{
							$this->loadModel("User");
							$this->loadModel("Video");
							$this->loadModel("VideoLike");
							$this->loadModel("VideoComment");
							$this->loadModel("Follower");
							$this->loadModel("Following");
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$userId = $this->request->getData('user_id');
							$userDetails = $this->User->getUserDetailsFromIDContain($userId);
							$userDetails['User']['videos_count'] = $this->Video->getUserVideosCount($userId);
							$userDetails['User']['followers_count'] = $this->Follower->countFollowers($userId);
							$userDetails['User']['following_count'] = $this->Follower->countFollowing($userId);
							$wallet = $this->walletTotal($userId);
							$userDetails['User']['wallet'] = $wallet['total'] ?? 0;
							$output = ['code' => 200, 'msg' => $userDetails];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}

						public function showPopularHashtags(): ResponseInterface
						{
							$this->loadModel("HashtagVideo");
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$hashtags = $this->HashtagVideo->getHastagsWhichHasGreaterNoOfVideosAdmin();
							$newArray = [];
							foreach ($hashtags as $key => $hashtag) {
								$count = $this->HashtagVideo->countHashtagVideos($hashtag['Hashtag']['id']);
								if ($count > 0) {
									$newArray[$key]["Hashtag"] = $hashtag['Hashtag'];
									$newArray[$key]["Hashtag"]['views'] = $hashtag[0]['total_views'];
									$newArray[$key]["Hashtag"]['videos'] = $count;
								}
							}
							if (empty($newArray)) {
								return $this->response->withStatus(204);
							}
							$output = ['code' => 200, 'msg' => $newArray];
							return $this->response->withType('application/json')->withStringBody(json_encode($output));
						}

						public function changePassword(): ResponseInterface
						{
							$this->loadModel('User');
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$userId = $this->request->getData('user_id');
							$newPassword = $this->request->getData('password');
							$user = $this->User->get($userId);
							$user = $this->User->patchEntity($user, ['password' => $newPassword]);
							if ($this->User->save($user)) {
								$userInfo = $this->User->getUserDetailsFromID($userId);
								$result = ['code' => 200, 'msg' => $userInfo];
								return $this->response->withType('application/json')->withStringBody(json_encode($result));
							}
							$error = ['code' => 500, 'msg' => 'Unable to save password'];
							return $this->response->withType('application/json')->withStringBody(json_encode($error))->withStatus(500);
						}

						public function deleteReportReason(): ResponseInterface
						{
							$this->loadModel('ReportReason');
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$id = $this->request->getData('id');
							try {
								$entity = $this->ReportReason->get($id);
								$this->ReportReason->delete($entity);
								$output = ['code' => 200, 'msg' => 'deleted'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} catch (\Exception $e) {
								$output = ['code' => 404, 'msg' => 'Invalid id: Does not exist'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output))->withStatus(404);
							}
						}

						public function deleteUser(): ResponseInterface
						{
							$this->loadModel('User');
							$this->loadModel('Follower');
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$userId = $this->request->getData('user_id');
							try {
								$user = $this->User->get($userId);
								$this->User->delete($user, true);
								$this->Follower->deleteFollowerAgainstUserID($userId);
								$this->firebase->deleteUser($user->auth_token);
								Cache::delete("user_details_" . $userId);
								$output = ['code' => 200, 'msg' => 'deleted'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} catch (\Exception $e) {
								$output = ['code' => 404, 'msg' => 'Invalid id: Does not exist'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output))->withStatus(404);
							}
						}

						public function deleteAdmin(): ResponseInterface
						{
							$this->loadModel('Admin');
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$userId = $this->request->getData('user_id');
							try {
								$admin = $this->Admin->get($userId);
								$this->Admin->delete($admin);
								$output = ['code' => 200, 'msg' => 'deleted'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							} catch (\Exception $e) {
								$output = ['code' => 404, 'msg' => 'Invalid id: Does not exist'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output))->withStatus(404);
							}
						}

						public function addAdminUser(): ResponseInterface
						{
							$this->loadModel('Admin');
							if (!$this->request->is('post')) {
								return $this->response->withStatus(405);
							}
							$data = [
								'email'      => $this->request->getData('email'),
								'password'   => $this->request->getData('password'),
								'first_name' => $this->request->getData('first_name'),
								'last_name'  => $this->request->getData('last_name'),
								'role'       => $this->request->getData('role'),
								'created'    => date('Y-m-d H:i:s')
							];
							if (empty($data['email']) || empty($data['password'])) {
								$output = ['code' => 400, 'msg' => 'Email and password are required'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output))->withStatus(400);
							}
							$count = $this->Admin->isEmailAlreadyExist($data['email']);
							if ($count) {
								$output = ['code' => 409, 'msg' => 'Email already exists'];
								return $this->response->withType('application/json')->withStringBody(json_encode($output))->withStatus(409);
							}
							$admin = $this->Admin->newEntity($data);
							if ($this->Admin->save($admin)) {
								$userDetails = $this->Admin->getUserDetailsFromID($admin->id);
								$output = ['code' => 200, 'msg' => $userDetails];
								return $this->response->withType('application/json')->withStringBody(json_encode($output));
							}
							$output = ['code' => 500, 'msg' => 'Unable to save admin user'];
							return $this->response->withType('application/json')->withStringBody(json_encode($output))->withStatus(500);
						}
    
						public function editAdminUser(): void
						{
							$this->loadModel('Admins');
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								$user = $this->Admins->get($data['id']);
								$user = $this->Admins->patchEntity($user, [
									'email' => $data['email'],
									'first_name' => $data['first_name'],
									'last_name' => $data['last_name'],
									'role' => $data['role'],
									'created' => date('Y-m-d H:i:s')
								]);
								if ($this->Admins->save($user)) {
									$this->set([
										'code' => 200,
										'msg' => $user,
										'_serialize' => ['code', 'msg']
									]);
								} else {
									Message::EMPTYDATA();
								}
							}
						}

						public function showAdminUsers(): void
						{
							$this->loadModel('Admins');
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								if (!empty($data['id'])) {
									$details = $this->Admins->get($data['id']);
								} else {
									$details = $this->Admins->find()->all();
								}
								if (!$details->isEmpty()) {
									$this->set([
										'code' => 200,
										'msg' => $details,
										'_serialize' => ['code', 'msg']
									]);
								} else {
									Message::EMPTYDATA();
								}
							}
						}

						public function showAllUsers(): void
						{
							$this->loadModel('Users');
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								$starting_point = $data['starting_point'] ?? 0;
								$business = $data['business'] ?? 0;
								$users = $this->Users->getAllUsers($business, $starting_point);
								$this->set([
									'code' => 200,
									'msg' => $users,
									'_serialize' => ['code', 'msg']
								]);
							}
						}

						public function jsonFileForm(): void
						{
							$this->autoRender = true;
						}

						public function importJsonFile(): void
						{
							$this->autoRender = true;
							if (Configure::read('App.status') === 'demo') {
								$this->set([
									'code' => 201,
									'msg' => 'you cannot upload a file using demo',
									'_serialize' => ['code', 'msg']
								]);
								return;
							}
							$filePath = Utility::uploadOriginalVideoFileIntoTemporaryFolder('json');
							if ($filePath) {
								$this->set([
									'code' => 200,
									'msg' => $filePath,
									'_serialize' => ['code', 'msg']
								]);
							} else {
								$this->set([
									'code' => 201,
									'msg' => 'something went wrong in uploading a file.',
									'_serialize' => ['code', 'msg']
								]);
							}
						}

						public function showJsonDatabaseFiles(): void
						{
							$files = glob(TEMP_UPLOADS_FOLDER_URI . '/*.json');
							$this->set([
								'code' => 200,
								'msg' => $files,
								'_serialize' => ['code', 'msg']
							]);
						}

						public function deleteJsonFile(): void
						{
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								$filePath = $data['file_path'];
								if (!file_exists($filePath)) {
									$this->set([
										'code' => 201,
										'msg' => 'file does not exist',
										'_serialize' => ['code', 'msg']
									]);
									return;
								}
								unlink($filePath);
								if (!file_exists($filePath)) {
									$this->set([
										'code' => 200,
										'msg' => 'deleted',
										'_serialize' => ['code', 'msg']
									]);
								} else {
									$this->set([
										'code' => 201,
										'msg' => 'something went wrong in deleting a file',
										'_serialize' => ['code', 'msg']
									]);
								}
							}
						}

						public function search(): void
						{
							$this->loadModel('Users');
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								$result = $this->Users->getSearchResultsAgainstType($data['keyword'], 0, $data['type']);
								if (!empty($result)) {
									$this->set([
										'code' => 200,
										'msg' => $result,
										'_serialize' => ['code', 'msg']
									]);
								} else {
									Message::EMPTYDATA();
								}
							}
						}

						public function changeAdminPassword(): void
						{
							$this->loadModel('Admins');
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								$user = $this->Admins->get($data['user_id']);
								$user = $this->Admins->patchEntity($user, ['password' => $data['password']]);
								if ($this->Admins->save($user)) {
									$this->set([
										'code' => 200,
										'msg' => $user,
										'_serialize' => ['code', 'msg']
									]);
								} else {
									Message::DATASAVEERROR();
								}
							} else {
								Message::INCORRECTPASSWORD();
							}
						}

						public function currentAdminChangePassword(): void
						{
							$this->loadModel('Admins');
							if ($this->request->is('post')) {
								$data = $this->request->getData();
								$user = $this->Admins->get($data['user_id']);
								if ($this->Admins->verifyPassword($user->email, $data['old_password'])) {
									$user = $this->Admins->patchEntity($user, ['password' => $data['new_password']]);
									if ($this->Admins->save($user)) {
										$this->set([
											'code' => 200,
											'msg' => $user,
											'_serialize' => ['code', 'msg']
										]);
									} else {
										Message::DATASAVEERROR();
									}
								} else {
									Message::INCORRECTPASSWORD();
								}
							}
						}
}
