<?php
declare(strict_types=1);
namespace Stripe\Model\Table;

use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\Validation\Validation;

class StripeCustomersTable extends Table {
	public string $path = '/customers';
	public array $_schema = [
		'id' => ['type' => 'integer', 'length' => 12],
		'card' => ['type' => 'string'],
		'number' => ['type' => 'string'],
		'exp_month' => ['type' => 'string', 'length' => 2],
		'exp_year' => ['type' => 'string', 'length' => 4],
		'cvc' => ['type' => 'string'],
		'name' => ['type' => 'string'],
		'address_line_1' => ['type' => 'string'],
		'address_line_2' => ['type' => 'string'],
		'address_zip' => ['type' => 'string'],
		'address_state' => ['type' => 'string'],
		'address_country' => ['type' => 'string'],
		'email' => ['type' => 'string'],
		'description' => ['type' => 'string'],
		'plan' => ['type' => 'string'],
		'trial_end' => ['type' => 'string'],
		'coupon' => ['type' => 'string']
	];
	public array $formatFields = [
		'card' => [
			'number',
			'exp_month',
			'exp_year',
			'cvc',
			'name',
			'address_line_1',
			'address_line_2',
			'address_zip',
			'address_state',
			'address_country'
		]
	];
	public function initialize(array $config): void {
		parent::initialize($config);
		$this->setTable('stripe_customers');
		$this->setDisplayField('id');
		$this->setPrimaryKey('id');
	}
	public function validationDefault(Validator $validator): Validator {
		$validator
			->notEmptyString('number', 'Please enter your credit card number.')
			->add('number', 'credit_card', [
				'rule' => function($value, $context) {
					return Validation::creditCard($value, ['visa', 'mc', 'amex', 'disc', 'jcb']);
				},
				'message' => 'Invalid credit card number.'
			])
			->notEmptyString('exp_month', 'Please enter your expiration month.')
			->add('exp_month', 'validMonth', [
				'rule' => function($value, $context) {
					return is_numeric($value) && (int)$value >= 1 && (int)$value <= 12;
				},
				'message' => 'Please enter a valid month.'
			])
			->notEmptyString('exp_year', 'Please enter your expiration year.')
			->add('exp_year', 'length', [
				'rule' => ['lengthBetween', 4, 4],
				'message' => 'Please enter a valid year.'
			])
			->notEmptyString('cvc', 'Please enter your CVC.')
			->add('cvc', 'numeric', [
				'rule' => 'numeric',
				'message' => 'Please enter a valid CVC.'
			])
			->add('address_zip', 'postal', [
				'rule' => ['postal', null, 'us'],
				'message' => 'Please enter a valid zipcode.'
			]);
		return $validator;
	}
}
