<?php
declare(strict_types=1);

namespace App\Controller;

use App\Lib\Utility;
use App\Lib\Message;
use App\Lib\CustomEmail;
use App\Lib\Firebase;
use App\Lib\Aws;
use App\Lib\Regular;
use Cake\Event\EventInterface;

class ApiController extends AppController
{
	public $aws;
	public $firebase;
	public $user_details = [];

	public function initialize(): void
	{
		parent::initialize();
		$this->autoRender = false;
		$this->viewBuilder()->disableAutoLayout();
		$this->firebase = Firebase::getInstance();
		$this->aws = Aws::getInstance();
		$this->user_details = [];
	}

	public function beforeFilter(EventInterface $event): void
	{
		parent::beforeFilter($event);
		if ($this->getRequest()->is('post')) {
			$json = (string)$this->getRequest()->getInput();
			$json_error = Utility::isJsonError($json);
			$request_uri = $this->getRequest()->getRequestTarget();
			$end_point = basename($request_uri);
			if (in_array($end_point, ['videoCompressorUpdate', 'getVideoDetection', 'snsNotification'], true)) {
				return;
			}
			$client_api_key = $this->getRequest()->getHeaderLine('Api-Key') ?: $this->getRequest()->getHeaderLine('API-KEY') ?: $this->getRequest()->getHeaderLine('api-key');
			if ($client_api_key !== '') {
				if ($client_api_key !== API_KEY) {
					Message::ACCESSRESTRICTED();
					exit;
				}
			} else {
				echo json_encode(['code' => 201, 'msg' => 'API KEY is missing']);
				exit;
			}
			Utility::logRequest($json, $end_point);
			$auth_token = $this->getRequest()->getHeaderLine('Auth-Token') ?: $this->getRequest()->getHeaderLine('AUTH-TOKEN') ?: $this->getRequest()->getHeaderLine('auth-token');
			if (strlen((string)$auth_token) > 5) {
				$this->loadModel('Users');
				$user_details = $this->Users->getUserDetailsAgainstAuthToken($auth_token);
				if (!empty($user_details)) {
					$this->user_details = $user_details;
				}
			}
			if ($json_error === false) {
				return;
			} else {
				$privacy_type = $this->getRequest()->getData('privacy_type');
				if (!empty($privacy_type)) {
				}
				return;
				echo json_encode(['code' => 202, 'msg' => $json_error]);
				exit;
			}
		} else {
			Message::METHODNOTALLOWED();
		}
	}

	public function index(): void
	{
		echo 'Congratulations!. You have configured your mobile api correctly';
	}
	
	public function registerUser(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('User');
		$this->loadModel('PushNotification');
		$this->loadModel('PrivacySetting');
		if ($this->getRequest()->is('post')) {
			$data = json_decode($this->getRequest()->getBody()->getContents(), true);
			$this->verifyParameters($data, ['auth_token','device_token']);
			$entity = $this->User->newEmptyEntity();
			$entity->created = date('Y-m-d H:i:s');
			$entity->device_token = $data['device_token'] ?? null;
			$entity->role = 'user';
			$authToken = $data['auth_token'];
			if (strlen($authToken) < 5) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'Something went wrong. Please try again']));
			}
			$firebase = $this->firebase->getUserDetailsAgainstUID($authToken);
			if (!$firebase) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'Invalid Credentials']));
			}
			$provider = $firebase->providerData[0]->providerId ?? null;
			$entity->register_with = $provider === 'password' ? 'email' : ($provider ?? 'fake');
			$entity->email = $firebase->email;
			$entity->phone = $firebase->phoneNumber;
			if (!empty($data['profile_pic'])) {
				$entity->profile_pic = $data['profile_pic'];
			}
			$entity->auth_token = $authToken;
			$entity->referral_code = Utility::getToken(9);
			if ($this->User->exists(['auth_token' => $authToken])) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'This account is already registered.']));
			}
			if (!empty($data['first_name'])) {
				$entity->first_name = $data['first_name'];
				$entity->last_name = $data['last_name'] ?? null;
			}
			if (!empty($data['dob'])) {
				$entity->dob = $data['dob'];
			}
			if (!empty($data['username'])) {
				if (preg_match('/[^a-z0-9_\-]/i', $data['username'])) {
					return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'invalid username']));
				}
				if ($this->User->exists(['username' => $data['username']])) {
					return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>"This username isn't available"]));
				}
				$entity->username = $data['username'];
			}
			if (!empty($entity->email) && strlen($entity->email) > 3 && $this->User->exists(['email' => $entity->email])) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'This account exists already, if you are the account owner, please login, if not, please use a different number or email.']));
			}
			if (!empty($entity->phone) && strlen($entity->phone) > 3 && $this->User->exists(['phone' => $entity->phone])) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'This account exists already, if you are the account owner, please login, if not, please use a different number or email.']));
			}
			if (!$this->User->save($entity)) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>500,'msg'=>'Data save error']));
			}
			$userId = $entity->id;
			if (!$this->PushNotification->exists(['id' => $userId])) {
				$notif = $this->PushNotification->newEmptyEntity();
				$notif->id = $userId;
				$notif->likes = 1;
				$notif->comments = 1;
				$notif->new_followers = 1;
				$notif->mentions = 1;
				$notif->direct_messages = 1;
				$notif->video_updates = 1;
				$this->PushNotification->save($notif);
			}
			if (!$this->PrivacySetting->exists(['id' => $userId])) {
				$ps = $this->PrivacySetting->newEmptyEntity();
				$ps->id = $userId;
				$ps->videos_download = 1;
				$ps->videos_repost = 1;
				$ps->direct_message = 'everyone';
				$ps->duet = 'everyone';
				$ps->liked_videos = 'me';
				$ps->video_comment = 'everyone';
				$this->PrivacySetting->save($ps);
			}
			$userDetails = $this->User->get($userId)->toArray();
			$userDetails['wallet'] = 0;
			$userDetails['comission_earned'] = 0;
			$userDetails['total_all_time_coins'] = 0;
			return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$userDetails]));
		}
	}

	public function reportRoom(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('ReportRoom');
		$this->loadModel('ReportReason');
		$this->loadModel('Room');
		if ($this->getRequest()->is('post')) {
			$data = $this->getRequest()->getData();
			$entity = $this->ReportRoom->newEmptyEntity();
			$entity->user_id = $data['user_id'];
			$entity->room_id = $data['room_id'];
			$entity->report_reason_id = $data['report_reason_id'];
			$entity->description = $data['description'];
			$entity->created = date('Y-m-d H:i:s');
			if ($this->ReportRoom->exists(['user_id' => $entity->user_id, 'room_id' => $entity->room_id])) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'Room has already been reported by you']));
			}
			$room = $this->Room->find()->where(['id' => $entity->room_id])->first();
			if (!$room) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'user not available']));
			}
			$reason = $this->ReportReason->find()->where(['id' => $entity->report_reason_id])->first();
			if ($reason) {
				$entity->report_reason_title = $reason->title;
			}
			if (!$this->ReportRoom->save($entity)) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>500,'msg'=>'Data save error']));
			}
			$details = $this->ReportRoom->get($entity->id);
			return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$details]));
		}
	}

	public function showFollowers(): \Psr\Http\Message\ResponseInterface
	{
		$this->loadModel('Follower');
		$this->loadModel('Video');
		$this->loadModel('VideoLike');
		if ($this->getRequest()->is('post')) {
			$data = $this->getRequest()->getData();
			$userId = $this->request->getAttribute('identity')->id ?? 0;
			$starting = $data['starting_point'] ?? 0;
			$targetUserId = $data['other_user_id'] ?? $userId;
			if ($starting < 1) {
				$cacheKey = 'followers_'.$targetUserId;
				$followers = Cache::read($cacheKey);
				if (!$followers) {
					$followers = $this->Follower->getUserFollowers($targetUserId, $starting);
					Cache::write($cacheKey, $followers);
				}
			} else {
				$followers = $this->Follower->getUserFollowers($targetUserId, $starting);
			}
			if (empty($followers)) {
				return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>204,'msg'=>[]]));
			}
			foreach ($followers as &$f) {
				$f['User'] = $f['FollowerList'];
				$person = $f['FollowerList']['id'];
				unset($f['FollowerList']);
				$this->populateUserStats($f, $userId, $person);
			}
			return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$followers]));
		}
	}

	public function joinRoom(): void
	{
		$this->loadModel('Rooms');
		$this->loadModel('RoomMembers');
		if (!$this->getRequest()->is('post')) {
			return;
		}
		$data = $this->getRequest()->getData();
		$identity = $this->request->getAttribute('identity');
		if (!$identity) {
			throw new \Cake\Http\Exception\UnauthorizedException();
		}
		$userId = $identity->get('id');
		$roomId = $data['room_id'];
		$moderator = $data['moderator'] ?? 0;
		$exists = $this->RoomMembers->exists(['room_id' => $roomId, 'user_id' => $userId]);
		if (!$exists) {
			$roomMember = $this->RoomMembers->newEmptyEntity();
			$roomMember = $this->RoomMembers->patchEntity($roomMember, [
				'user_id'    => $userId,
				'room_id'    => $roomId,
				'moderator'  => $moderator,
				'created'    => date('Y-m-d H:i:s')
			]);
			if ($this->RoomMembers->save($roomMember)) {
				$output = ['code' => 200, 'msg' => $roomMember];
			} else {
				$output = ['code' => 500, 'msg' => 'Unable to join room'];
			}
		} else {
			$output = ['code' => 201, 'msg' => 'You are already a member of this group'];
		}
		$this->autoRender = false;
		$this->response = $this->response
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function showRoomDetail(): void
	{
		$this->loadModel('Rooms');
		$this->loadModel('RoomMembers');
		if (!$this->getRequest()->is('post')) {
			return;
		}
		$data = $this->getRequest()->getData();
		$roomId = $data['room_id'];
		$room    = $this->Rooms->get($roomId)->toArray();
		$member       = 0;
		$memberCount  = 0;
		if (!empty($data['user_id'])) {
			$userId = $data['user_id'];
			$exists = $this->RoomMembers->exists(['room_id' => $roomId, 'user_id' => $userId]);
			if ($exists) {
				$member      = 1;
				$memberCount = $this->RoomMembers->find()
					->where(['room_id' => $roomId])
					->count();
			}
		}
		$room['member']       = $member;
		$room['member_count'] = $memberCount;
		$output = ['code' => 200, 'msg' => ['Room' => $room]];
		$this->autoRender = false;
		$this->response = $this->response
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function leaveRoom(): void
	{
		$this->loadModel('Rooms');
		$this->loadModel('RoomMembers');
		if (!$this->getRequest()->is('post')) {
			return;
		}
		$data     = $this->getRequest()->getData();
		$identity = $this->request->getAttribute('identity');
		if (!$identity) {
			throw new \Cake\Http\Exception\UnauthorizedException();
		}
		$userId = $identity->get('id');
		$roomId = $data['room_id'];
		$member = $this->RoomMembers->find()
			->where(['room_id' => $roomId, 'user_id' => $userId])
			->first();
		if ($member) {
			$this->RoomMembers->delete($member);
			$output = ['code' => 200, 'msg' => 'removed member'];
		} else {
			throw new \Cake\Http\Exception\NotFoundException();
		}
		$this->autoRender = false;
		$this->response = $this->response
			->withType('application/json')
			->withStringBody(json_encode($output));
	}

	public function showUserJoinedRooms(): void
	{
		$this->loadModel('RoomMember');
		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$user_id = 0;
			$user_details = $this->user_details;

			if (!empty($user_details)) {
				$user_id = $user_details['User']['id'];
			}

			$details = $this->RoomMember->getUserRooms($user_id);

			if (!empty($details)) {
				foreach ($details as $key => $detail) {
					$room_id = $detail['RoomMember']['room_id'];
					$moderators = $this->RoomMember->getRoomModerators($room_id);
					$details[$key]['Room']['Moderators'] = $moderators;
				}
				$output = [
					'code' => 200,
					'msg' => $details
				];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			} else {
				Message::EMPTYDATA();
				return;
			}
		}
	}

	public function showProfileVisitors(): void
	{
		$this->loadModel('ProfileVisit');
		$this->loadModel('Follower');
		$this->loadModel('Video');

		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$created = date('Y-m-d H:i:s');
			$user_details = $this->user_details;

			if (empty($user_details)) {
				Message::UNAUTHORIZED();
				return;
			}

			$user_id = $user_details['User']['id'];
			$last_date = date('Y-m-d', strtotime("-30 days", strtotime($created)));
			$this->ProfileVisit->updateReadCount($user_id);
			$starting_point = $data['starting_point'] ?? 0;

			$visitors = $this->ProfileVisit->getProfileVisitors($user_id, $last_date, $starting_point);
			if (!empty($visitors)) {
				$new_data = [];

				foreach ($visitors as $key => $follow) {
					$person_user_id = $follow['Visitor']['id'];
					$new_data[$key]['User'] = $follow['Visitor'];
					$this->populateUserStats($new_data[$key], $user_id, $person_user_id);
					$visitors[$key]['Visitor'] = $new_data[$key]['User'];
					$visitors[$key]['User'] = $new_data[$key]['User'];
				}

				$output = [
					'code' => 200,
					'msg' => $visitors
				];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			} else {
				Message::EMPTYDATA();
				return;
			}
		}
	}

	public function deleteSound(): void
	{
		$this->loadModel('Sound');
		$this->loadModel('Video');

		if ($this->request->is('post')) {
			$data = $this->request->getData();
			$sound_id = $data['sound_id'] ?? null;

			if (!$sound_id) {
				$output = [
					'code' => 201,
					'msg' => 'Sound ID is missing'
				];
				$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				return;
			}

			$details = $this->Sound->getDetails($sound_id);

			if (!empty($details)) {
				$audio_url = $details['Sound']['audio'];

				if (str_starts_with($audio_url, 'http')) {
					if (method_exists($this->aws, 'deleteObjectS3')) {
						$result1 = $this->aws->deleteObjectS3($audio_url);
						if ($result1) {
							$code = 200;
							$msg = "deleted successfully";
						} else {
							$code = 201;
							$msg = "something went wrong in deleting the file from the cdn";
						}
					}
				} else {
					Utility::unlinkFile($audio_url);
					$code = 200;
					$msg = "successfully deleted";
				}

				$this->Sound->delete($sound_id);
				$all_videos = $this->Video->getAllVideosAgainstSoundID($sound_id);
				if (!empty($all_videos)) {
					$video_ids = [];
					foreach ($all_videos as $val) {
						$video_ids[] = $val['Video']['id'];
					}
					$this->Video->updateSoundIDs($video_ids);
				}

				$output = [
					'code' => 200,
					'msg' => 'deleted'
				];
			} else {
				$output = [
					'code' => 201,
					'msg' => 'Invalid id: Do not exist'
				];
			}

			$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
			return;
		}
	}

	public function showFollowing(): Response
		{
			$this->loadModel('Followers');
			$this->loadModel('Videos');
			$this->loadModel('VideoLikes');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$userId = $this->Authentication->getIdentity()->getIdentifier() ?? 0;
				$startingPoint = $data['starting_point'] ?? 0;
				$targetUserId = $data['other_user_id'] ?? $userId;
				if ($startingPoint < 1) {
					$cacheKey = 'following_' . $targetUserId;
					$following = Cache::read($cacheKey);
					if ($following === null) {
						$following = $this->Followers->getUserFollowing($targetUserId, $startingPoint);
						Cache::write($cacheKey, $following);
					}
				} else {
					$following = $this->Followers->getUserFollowing($targetUserId, $startingPoint);
				}
				if (!empty($following)) {
					foreach ($following as $key => $follow) {
						$personUserId = $follow['FollowingList']['id'];
						$following[$key]['User'] = $follow['FollowingList'];
						unset($following[$key]['FollowingList']);
						$this->populateUserStats($following[$key], $userId, $personUserId);
					}
					$payload = ['code' => 200, 'msg' => $following];
				} else {
					$payload = Message::EMPTYDATA();
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function deleteVideo(): Response
		{
			$this->loadModel('Videos');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$this->verifyParameters($data, ['video_id']);
				$user = $this->Authentication->getIdentity();
				if (!$user) {
					return Message::UNAUTHORIZED();
				}
				$videoId = (int)$data['video_id'];
				$details = $this->Videos->getOnlyVideoDetailsAgainstVideoAndUserID($user->getIdentifier(), $videoId);
				if (!empty($details)) {
					[$videoUrl, $thumUrl, $gifUrl] = [
						$details['video'],
						$details['thum'],
						$details['gif']
					];
					if (str_starts_with($videoUrl, 'http')) {
						if (method_exists($this->Aws, 'deleteObjectS3')) {
							$this->Aws->deleteObjectS3($videoUrl);
							$this->Aws->deleteObjectS3($thumUrl);
							$this->Aws->deleteObjectS3($gifUrl);
							$msg = 'deleted successfully';
						}
					} else {
						foreach ([$videoUrl, $thumUrl, $gifUrl] as $file) {
							if (file_exists($file)) {
								unlink($file);
							}
						}
						$msg = 'successfully deleted';
					}
				} else {
					$msg = 'video has been already deleted';
				}
				Cache::delete("video_details_{$videoId}");
				$this->Videos->delete($this->Videos->get($videoId));
				$payload = ['code' => 200, 'msg' => $msg];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function followUser(): Response
		{
			$this->loadModel('Users');
			$this->loadModel('Notifications');
			$this->loadModel('PushNotifications');
			$this->loadModel('Followers');
			if (!$this->request->is('post')) {
				return $this->response->withStatus(405);
			}
			$data = json_decode($this->request->getInput(), true);
			$this->verifyParameters($data, ['receiver_id']);
			$user = $this->Authentication->getIdentity();
			if (!$user) {
				return Message::UNAUTHORIZED();
			}
			$senderId = $user->getIdentifier();
			$receiverId = (int)$data['receiver_id'];
			$now = date('Y-m-d H:i:s');
			Cache::delete("following_{$senderId}");
			Cache::delete("followers_{$receiverId}");
			$exists = $this->Followers->ifFollowing($senderId, $receiverId);
			$receiver = $this->getUserDetailsFromCache($receiverId);
			if (empty($receiver)) {
				return Message::DONOTEXIST('User');
			}
			if (empty($exists)) {
				$this->Followers->save(['sender_id' => $senderId, 'receiver_id' => $receiverId, 'created' => $now]);
				$msg = $user->get('username') . ' started following you';
				if (!empty($receiver['device_token']) && $receiver['PushNotification']['new_followers'] > 0) {
					$this->firebase->sendPushNotification([
						'to' => $receiver['device_token'],
						'notification' => ['title' => $msg, 'type' => 'follow', 'receiver_id' => $receiverId, 'sender_id' => $senderId],
						'data' => ['title' => $msg, 'type' => 'follow']
					]);
					$this->Notifications->save(['sender_id' => $senderId, 'receiver_id' => $receiverId, 'type' => 'following', 'string' => $msg, 'created' => $now]);
					Cache::delete("notifications_{$receiverId}");
				}
				$this->populateUserStats($receiver, $senderId, $receiverId);
			} else {
				$this->Followers->delete($exists['id']);
				$this->populateUserStats($receiver, $senderId, $receiverId);
			}
			$payload = ['code' => 200, 'msg' => $receiver];
			return $this->response
				->withType('application/json')
				->withStringBody(json_encode($payload));
		}

		private function sendPushNotification(array $sender, array $receiver, string $msg, string $type): void
		{
			$this->loadModel('Notifications');
			$this->loadModel('PushNotifications');
			$senderId = $sender['id'];
			$receiverId = $receiver['id'];
			$now = date('Y-m-d H:i:s');
			if (strlen($receiver['device_token'] ?? '') > 8) {
				$tokenModel = $this->PushNotifications->get($receiverId);
				$countKey = $type === 'follow' ? 'new_followers' : '';
				if ($tokenModel->{$countKey} > 0) {
					$this->firebase->sendPushNotification(['to' => $receiver['device_token'], 'notification' => ['title' => $msg, 'type' => $type]]);
				}
				$this->Notifications->save(['sender_id' => $senderId, 'receiver_id' => $receiverId, 'type' => $type, 'string' => $msg, 'created' => $now]);
				Cache::delete("notifications_{$receiverId}");
			}
		}

		public function registerDevice(): Response
		{
			$this->loadModel('Devices');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$key = $data['key'] ?? null;
				$now = date('Y-m-d H:i:s');
				if ($key === null) {
					return $this->response->withStatus(400);
				}
				$device = $this->Devices->findByKey($key)->first();
				if (!$device) {
					$device = $this->Devices->newEntity(['key' => $key, 'created' => $now]);
					$this->Devices->save($device);
				}
				$payload = ['code' => 200, 'msg' => $device];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function updatePushNotificationSettings(): Response
		{
			$this->loadModel('PushNotifications');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$user = $this->Authentication->getIdentity();
				if (!$user) {
					return Message::UNAUTHORIZED();
				}
				$settings = array_intersect_key($data, array_flip(['likes','comments','new_followers','mentions','direct_messages','video_updates']));
				$entity = $this->PushNotifications->get($user->getIdentifier());
				$entity = $this->PushNotifications->patchEntity($entity, $settings);
				$this->PushNotifications->save($entity);
				$payload = ['code' => 200, 'msg' => $entity];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function addPrivacySetting(): Response
		{
			$this->loadModel('PrivacySettings');
			$this->loadModel('Users');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$userId = $data['user_id'] ?? $this->Authentication->getIdentity()->getIdentifier();
				if (!$userId) {
					return Message::EMPTYDATA();
				}
				$fields = array_intersect_key($data, array_flip(['videos_download','order_history','direct_message','duet','liked_videos','video_comment']));
				$entity = $this->PrivacySettings->get($userId);
				$entity = $this->PrivacySettings->patchEntity($entity, $fields);
				$this->PrivacySettings->save($entity);
				$payload = ['code' => 200, 'msg' => $entity];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function followerNotification(): Response
		{
			$this->loadModel('Followers');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$exists = $this->Followers->ifFollowing((int)$data['sender_id'], (int)$data['receiver_id']);
				if ($exists) {
					$entity = $this->Followers->get($exists['id']);
					$entity->notification = $data['notification'];
					$this->Followers->save($entity);
					$payload = ['code' => 200, 'msg' => $entity];
				} else {
					$payload = Message::EMPTYDATA();
				}
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function showVideoCompression(): Response
		{
			if ($this->request->is('post')) {
				$payload = ['code' => 200, 'msg' => VIDEO_COMPRESSION];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function showVideoStats(): Response
		{
			$this->loadModel('Videos');
			$this->loadModel('VideoLikes');
			$this->loadModel('VideoFavourites');
			$this->loadModel('VideoComments');
			$this->loadModel('Followers');
			$this->loadModel('RepostVideos');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$video = $this->Videos->getOnlyVideoDetails((int)$data['video_id']);
				$userId = (int)$data['user_id'];
				$video['like'] = $this->VideoLikes->exists(['user_id' => $userId, 'video_id' => $video['id']]) ? 1 : 0;
				$video['favourite'] = $this->VideoFavourites->exists(['user_id' => $userId, 'video_id' => $video['id']]) ? 1 : 0;
				$video['repost'] = $this->RepostVideos->exists(['user_id' => $userId, 'video_id' => $video['id']]) ? 1 : 0;
				$follower = $this->Followers->ifFollowing($userId, $video['user_id']);
				$following = $this->Followers->ifFollowing($video['user_id'], $userId);
				$video['button'] = match (true) {
					$follower && $following => 'Friends',
					$follower => 'following',
					$following => 'follow back',
					default => 'follow'
				};
				$video['favourite_count'] = $this->VideoFavourites->find()->where(['video_id' => $video['id']])->count();
				$video['like_count'] = $this->VideoLikes->countLikes($video['id']);
				$video['comment_count'] = $this->VideoComments->countComments($video['id']);
				$payload = ['code' => 200, 'msg' => $video];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function showAllPromotions(): Response
		{
			$this->loadModel('Promotions');
			$this->loadModel('VideoWatches');
			$this->loadModel('VideoLikes');
			$this->loadModel('Followers');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$userId = (int)$data['user_id'];
				$days = $data['days'];
				$end = new \DateTime();
				$start = match ($days) {
					7 => (clone $end)->modify('-7 days'),
					28 => (clone $end)->modify('-28 days'),
					60 => (clone $end)->modify('-60 days'),
					default => new \DateTime($data['start_datetime'] ?? $end->format('Y-m-d H:i:s')),
				};
				$promos = $this->Promotions->getUserPromotions($userId);
				$stats = ['coins_spent' => 0, 'total_video_views' => 0, 'total_destination_tap' => 0, 'total_video_likes' => 0];
				foreach ($promos as $promo) {
					$videoId = $promo['video_id'];
					$stats['coins_spent'] += $promo['coin'];
					$views = $this->VideoWatches->countWatchVideos($videoId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
					$likes = $this->VideoLikes->countLikesBetweenDatetime($videoId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
					$promo['views'] = $views;
					$stats['total_video_views'] += $views;
					$stats['total_destination_tap'] += $promo['destination_tap'];
					$stats['total_video_likes'] += $likes;
				}
				$payload = [
					'code' => 200,
					'msg' => [
						'Promotions' => $promos,
						'Stats' => array_merge($stats, [
							'starting_date' => $start->format('M d'),
							'ending_date' => $end->format('M d')
						])
					]
				];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function showAnalytics(): Response
		{
			$this->loadModel('Videos');
			$this->loadModel('VideoWatches');
			$this->loadModel('VideoLikes');
			$this->loadModel('Followers');
			$this->loadModel('VideoComments');
			$this->loadModel('ProfileVisits');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$userId = (int)$data['user_id'];
				$start = new \DateTime($data['start_datetime']);
				$end = new \DateTime($data['end_datetime']);
				$countries = $this->Followers->countFollowersByCountry($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$cities = $this->Followers->countFollowersByCity($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$videos = $this->Videos->getAllUserVideos($userId);
				$views = $this->VideoWatches->countWatchVideos(array_column($videos, 'id'), $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$likes = $this->VideoLikes->countLikesBetweenDatetime(array_column($videos, 'id'), $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$comments = $this->VideoComments->countCommentsBetweenDates(array_column($videos, 'id'), $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$profileVisits = $this->ProfileVisits->getProfileViewsBetweenDates($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$totalFollowers = $this->Followers->countFollowersBetweenDatetime($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$maleFollowers = $this->Followers->countFollowersByGender($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'), 'male');
				$femaleFollowers = $this->Followers->countFollowersByGender($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'), 'female');
				$ageGroups = $this->Followers->getFollowersByAge($userId, $start->format('Y-m-d H:i:s'), $end->format('Y-m-d H:i:s'));
				$dates = new \DatePeriod($start, new \DateInterval('P1D'), (clone $end)->modify('+1 day'));
				$followersGraph = [];
				$viewsGraph = [];
				foreach ($dates as $date) {
					$dateStr = $date->format('Y-m-d');
					$countF = $this->Followers->countOnDate($userId, $dateStr);
					$countV = $this->VideoWatches->countOnDate(array_column($videos, 'id'), $dateStr);
					$followersGraph[] = ['date' => $dateStr, 'count' => $countF];
					$viewsGraph[] = ['date' => $dateStr, 'count' => $countV];
				}
				$payload = [
					'code' => 200,
					'msg' => [
						'Stats' => [
							'starting_date' => $start->format('M d'),
							'ending_date' => $end->format('M d'),
							'total_video_comments' => $comments,
							'total_video_views' => $views,
							'total_profile_visits' => $profileVisits,
							'total_video_likes' => $likes,
							'total_followers' => $totalFollowers,
							'total_male_followers' => $maleFollowers,
							'total_female_followers' => $femaleFollowers,
							'age_group' => $ageGroups,
							'video_views_graph' => $viewsGraph,
							'followers_graph' => $followersGraph,
							'followers_country' => $countries,
							'followers_city' => $cities
						]
					]
				];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function showDeviceDetail(): Response
		{
			$this->loadModel('Devices');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$device = $this->Devices->findByKey($data['key'] ?? '')->first();
				if ($device) {
					$payload = ['code' => 200, 'msg' => $device];
				} else {
					$payload = Message::EMPTYDATA();
				}
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function showProductDetailold(): Response
		{
			$this->loadModel('VideoProducts');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$productId = (int)$data['product_id'];
				$videoId = $data['video_id'] ?? null;
				$detail = $this->VideoProducts->ifExist(['product_id' => $productId, 'video_id' => $videoId]);
				if ($detail && $videoId) {
					$updated = new \DateTime($detail['updated_at']);
					if ((new \DateTime())->diff($updated)->days < 15) {
						$json = Mealme::getSingleProductDetail($productId);
						$this->VideoProducts->save($this->VideoProducts->patchEntity($detail, ['product_mealme_json' => $json, 'updated_at' => date('Y-m-d H:i:s')]));
					}
					$payload = ['code' => 200, 'msg' => $detail];
				} elseif (!$videoId) {
					$json = Mealme::getSingleProductDetail($productId);
					$payload = ['code' => 200, 'msg' => json_decode($json, true)];
				} else {
					$payload = Message::EMPTYDATA();
				}
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}

		public function test11(): void
		{
			$this->loadModel('VideoProducts');
			$this->loadModel('Stores');
			$details = $this->VideoProducts->find()->toArray();
			$addr = [
				'lat' => '37.3266047',
				'long' => '-121.8845012',
				'zip' => '11355',
				'state' => 'CA',
				'city' => 'Santa Clara',
				'street' => 'Clyde Avenue',
				'street_num' => '3120'
			];
			foreach ($details as $d) {
				$json = Mealme::getSingleProductDetail($d['product_id'], $addr, 1, 'false');
				$data = json_decode($json, true);
				if (!isset($data['error'])) {
					$entity = $this->VideoProducts->get($d['id']);
					$entity->product_mealme_json = $json;
					$entity->updated_at = date('Y-m-d H:i:s');
					$this->VideoProducts->save($entity);
				}
			}
		}

		public function removeCountryCode(string $phoneNumber): string
		{
			return str_starts_with($phoneNumber, '+1') ? substr($phoneNumber, 2) : $phoneNumber;
		}

		public function showProductDetailSystem(): Response
		{
			$this->loadModel('Products');
			$this->loadModel('ProductFavourites');
			$this->loadModel('ProductRatings');
			$this->loadModel('OrderProducts');
			$this->loadModel('DeliveryAddresses');
			if ($this->request->is('post')) {
				$data = json_decode($this->request->getInput(), true);
				$product = $this->Products->getDetailsWithAttributes((int)$data['product_id']);
				if (!$product) {
					return Message::EMPTYDATA();
				}
				$delivery = $this->DeliveryAddresses->getUserDefault((int)$data['user_id']);
				$fav = $this->ProductFavourites->ifProductFavourite((int)$data['user_id'], $product['id']);
				$ratings = $this->ProductRatings->getAvgRatings($product['id']);
				$sold = $this->OrderProducts->countSoldProducts($product['id']);
				$product['ProductFavourite']['favourite'] = $fav ? '1' : '0';
				$product['ProductFavourite']['favourite_count'] = $this->ProductFavourites->find()->where(['product_id' => $product['id']])->count();
				if ($delivery) {
					$product['Shipping'] = $delivery;
				}
				$product['TotalRatings'] = ['avg' => $ratings['average'] ?? 0, 'totalRatings' => $ratings['total_ratings'] ?? 0];
				$product['sold'] = $sold;
				$this->Products->saveField('view', $product['view'] + 1);
				$payload = ['code' => 200, 'msg' => $product];
				return $this->response->withType('application/json')->withStringBody(json_encode($payload));
			}
			return $this->response->withStatus(405);
		}
    
			<?php
			declare(strict_types=1);

			use Cake\Http\Response;
			use Psr\Http\Message\ResponseInterface;

			public function addFakeUser(string $storename, string $street, string $photo): array
			{
				$this->loadModel('Users');
				$this->loadModel('PushNotifications');
				$this->loadModel('PrivacySettings');
				$email = Utility::generateRandomEmail();
				$password = $email;
				$role = 'user';
				if (strpos($storename, ' ') !== false) {
					[$first_name, $last_name] = explode(' ', $storename, 2);
				} else {
					$first_name = $storename;
					$last_name = '';
				}
				$created = date('Y-m-d H:i:s');
				$unique_id = uniqid();
				$four_integers = substr(preg_replace('/\D/', '', $unique_id), 0, 4);
				$username = preg_replace('/[^A-Za-z0-9]/', '', $storename . $street);
				$user = [
					'email' => $email,
					'password' => $password,
					'first_name' => $first_name,
					'last_name' => $last_name,
					'username' => $username,
					'business' => 1,
					'role' => $role,
					'profile_pic' => $photo,
					'profile_pic_small' => $photo,
					'created' => $created
				];
				if ($this->Users->emailExists($email)) {
					$user['email'] = Utility::generateRandomEmail();
				}
				$entity = $this->Users->newEntity($user);
				if ($entity->getErrors()) {
					throw new \RuntimeException('Unable to save user');
				}
				$saved = $this->Users->save($entity);
				if (!$saved) {
					throw new \RuntimeException('Unable to save user');
				}
				$user_id = $saved->id;
				$notification = [
					'id' => $user_id,
					'likes' => 1,
					'comments' => 1,
					'new_followers' => 1,
					'mentions' => 1,
					'direct_messages' => 1,
					'video_updates' => 1
				];
				$this->PushNotifications->save($this->PushNotifications->newEntity($notification));
				$settings = [
					'id' => $user_id,
					'videos_download' => 1,
					'videos_repost' => 1,
					'direct_message' => 'everyone',
					'duet' => 'everyone',
					'liked_videos' => 'me',
					'video_comment' => 'everyone'
				];
				$this->PrivacySettings->save($this->PrivacySettings->newEntity($settings));
				return $this->Users->get($user_id)->toArray();
			}

			public function showRegisteredContacts(): ResponseInterface
			{
				$this->loadModel('Users');
				$this->loadModel('Followers');
				$data = $this->getRequest()->getData();
				$user_id = (int)($data['user_id'] ?? 0);
				$output = ['code' => 400, 'msg' => []];
				if (!empty($data['phone_numbers'])) {
					$matches = [];
					foreach ($data['phone_numbers'] as $val) {
						if ($this->Users->phoneExists((string)$val['phone'], $user_id)) {
							$matches[] = $val['phone'];
						}
					}
					if ($matches) {
						$output = ['code' => 200, 'msg' => $matches];
					}
				} elseif (!empty($data['facebook_ids'])) {
					$fb_ids = array_column($data['facebook_ids'], 'fb_id');
					$users = $this->Users->findByFacebookIds($fb_ids, $user_id);
					foreach ($users as &$u) {
						$pid = $u['id'];
						$u['follower_count'] = $this->Followers->countFollowers($pid);
						$u['video_count'] = $this->Videos->countByUser($pid);
						$f1 = $this->Followers->isFollowing($user_id, $pid);
						$f2 = $this->Followers->isFollowing($pid, $user_id);
						$u['button'] = $f1 && $f2 ? 'Friends' : ($f1 ? 'Following' : ($f2 ? 'Follow Back' : 'Follow'));
					}
					if ($users) {
						$output = ['code' => 200, 'msg' => $users];
					}
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showPromotedProducts(): ResponseInterface
			{
				$this->loadModel('Products');
				$this->loadModel('ProductRatings');
				$this->loadModel('OrderProducts');
				$data = $this->getRequest()->getData();
				$start = (int)($data['starting_point'] ?? 0);
				$products = $this->Products->getPromoted($start);
				$output = ['code' => 400, 'msg' => []];
				if ($products) {
					foreach ($products as &$p) {
						$pid = $p['id'];
						$rating = $this->ProductRatings->averageFor($pid);
						$sold = $this->OrderProducts->countSold($pid);
						$p['sold'] = $sold;
						$p['TotalRatings'] = ['avg' => $rating['avg'] ?? 0, 'totalRatings' => $rating['count'] ?? 0];
					}
					$output = ['code' => 200, 'msg' => $products];
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showTopViewedProducts(): ResponseInterface
			{
				$this->loadModel('Products');
				$this->loadModel('ProductRatings');
				$this->loadModel('OrderProducts');
				$data = $this->getRequest()->getData();
				$start = (int)($data['starting_point'] ?? 0);
				$products = $this->Products->getTopViewed($start);
				$output = ['code' => 400, 'msg' => []];
				if ($products) {
					foreach ($products as &$p) {
						$pid = $p['id'];
						$rating = $this->ProductRatings->averageFor($pid);
						$sold = $this->OrderProducts->countSold($pid);
						$p['sold'] = $sold;
						$p['TotalRatings'] = ['avg' => $rating['avg'] ?? 0, 'totalRatings' => $rating['count'] ?? 0];
					}
					$output = ['code' => 200, 'msg' => $products];
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showProductCategories(): ResponseInterface
			{
				$this->loadModel('Categories');
				$data = $this->getRequest()->getData();
				$start = (int)($data['starting_point'] ?? 0);
				$cats = $this->Categories->mostOccurring($start);
				$output = ['code' => $cats ? 200 : 400, 'msg' => $cats];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showOnlyProductCategories(): ResponseInterface
			{
				$this->loadModel('Categories');
				$data = $this->getRequest()->getData();
				$parent = (int)($data['parent_id'] ?? 0);
				$cats = $this->Categories->getByParent($parent);
				$output = ['code' => $cats ? 200 : 400, 'msg' => $cats];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showCategories(): ResponseInterface
			{
				$this->loadModel('Categories');
				$data = $this->getRequest()->getData();
				$parent = (int)($data['parent_id'] ?? 0);
				$cacheKey = 'categories';
				$cats = Cache::read($cacheKey);
				if ($cats === false) {
					$cats = $this->Categories->getByParent($parent);
					Cache::write($cacheKey, $cats);
				}
				$output = ['code' => $cats ? 200 : 400, 'msg' => $cats];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showProducts(): ResponseInterface
			{
				$this->loadModel('Products');
				$user = $this->request->getAttribute('identity');
				if (!$user) {
					return $this->response->withStatus(401);
				}
				$data = $this->getRequest()->getData();
				$start = (int)($data['starting_point'] ?? 0);
				$prods = $this->Products->forUser($user->id, $start);
				$output = ['code' => $prods ? 200 : 400, 'msg' => $prods];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showProductsAgainstCategories(): ResponseInterface
			{
				$this->loadModel('Products');
				$data = $this->getRequest()->getData();
				$category_id = (int)($data['category_id'] ?? 0);
				$start = (int)($data['starting_point'] ?? 0);
				$prods = $this->Products->forCategory($category_id, $start);
				$output = ['code' => $prods ? 200 : 400, 'msg' => $prods];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showProductsAttributes(): ResponseInterface
			{
				$this->loadModel('ProductAttributes');
				$id = (int)$this->getRequest()->getData('id');
				$attrs = $this->ProductAttributes->get($id);
				$output = ['code' => $attrs ? 200 : 400, 'msg' => $attrs];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showProductsAttributeVariation(): ResponseInterface
			{
				$this->loadModel('ProductAttributeVariations');
				$id = (int)$this->getRequest()->getData('id');
				$vars = $this->ProductAttributeVariations->get($id);
				$output = ['code' => $vars ? 200 : 400, 'msg' => $vars];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function addProduct(): ResponseInterface
			{
				$this->loadModel('Products');
				$this->loadModel('ProductImages');
				$user = $this->request->getAttribute('identity');
				if (!$user) {
					return $this->response->withStatus(401);
				}
				$data = $this->getRequest()->getData();
				$product = [
					'user_id' => $user->id,
					'category_id' => $data['category_id'] ?? null,
					'title' => $data['title'] ?? '',
					'condition' => $data['condition'] ?? '',
					'delivery_method' => $data['delivery_method'] ?? '',
					'price' => $data['price'] ?? 0,
					'description' => $data['description'] ?? '',
					'status' => $data['status'] ?? '',
					'meetup_location_lat' => $data['meetup_location_lat'] ?? null,
					'meetup_location_long' => $data['meetup_location_long'] ?? null,
					'meetup_location_string' => ''
				];
				if ($product['meetup_location_lat'] && $product['meetup_location_long']) {
					$loc = Utility::getCountryCityProvinceFromLatLong($product['meetup_location_lat'], $product['meetup_location_long']);
					$product['meetup_location_string'] = $loc['location_string'] ?? '';
				}
				$entity = isset($data['id'])
					? $this->Products->patchEntity($this->Products->get($data['id']), $product)
					: $this->Products->newEntity($product);
				if ($entity->getErrors()) {
					return $this->response->withStatus(422);
				}
				$saved = $this->Products->save($entity);
				if (!$saved) {
					return $this->response->withStatus(500);
				}
				if (empty($data['id'])) {
					$images = [];
					foreach ($data['images'] ?? [] as $img) {
						$images[] = ['product_id' => $saved->id, 'image' => $img['url'], 'created' => date('Y-m-d H:i:s')];
					}
					if ($images) {
						$this->ProductImages->saveMany($this->ProductImages->newEntities($images));
					}
				}
				$details = $this->Products->get($saved->id, ['contain' => []]);
				$output = ['code' => 200, 'msg' => $details];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function addProductRating(): ResponseInterface
			{
				$this->loadModel('ProductRatings');
				$this->loadModel('Products');
				$data = $this->getRequest()->getData();
				$ratingData = [
					'user_id' => $data['user_id'],
					'product_id' => $data['product_id'],
					'star' => $data['star'],
					'comment' => $data['comment'],
					'order_id' => $data['order_id'],
					'created' => date('Y-m-d H:i:s')
				];
				$existing = $this->ProductRatings->find()
					->where($ratingData)
					->first();
				$entity = $existing
					? $this->ProductRatings->patchEntity($existing, $ratingData)
					: $this->ProductRatings->newEntity($ratingData);
				$saved = $this->ProductRatings->save($entity);
				if (!$saved) {
					return $this->response->withStatus(500);
				}
				$details = $this->ProductRatings->get($saved->id);
				$output = ['code' => 200, 'msg' => $details];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function verifyCoupon(): ResponseInterface
			{
				$this->loadModel('Coupons');
				$this->loadModel('CouponUsages');
				$data = $this->getRequest()->getData();
				$coupon = $this->Coupons->findByCode($data['coupon_code'])->first();
				$output = ['code' => 201, 'msg' => 'invalid coupon code'];
				if ($coupon) {
					$usedCount = $this->CouponUsages->countForCoupon($coupon->id);
					$alreadyUsed = $this->CouponUsages->isUsedBy($coupon->id, (int)$data['user_id']);
					if (!$alreadyUsed && $usedCount < $coupon->limit_users) {
						$output = ['code' => 200, 'msg' => $coupon];
					}
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showProductDetail(): ResponseInterface
			{
				$this->loadModel('Products');
				$this->loadModel('DeliveryAddresses');
				$this->loadModel('ProductFavourites');
				$this->loadModel('ProductRatings');
				$this->loadModel('OrderProducts');
				$data = $this->getRequest()->getData();
				$pid = (int)$data['product_id'];
				$user = $this->request->getAttribute('identity');
				$details = $this->Products->get($pid, ['contain' => ['ProductAttributes', 'ProductAttributeVariations']]);
				$output = ['code' => 400, 'msg' => []];
				if ($details) {
					$favs = $this->ProductFavourites->isFavourite($user->id, $pid);
					$favCount = $this->ProductFavourites->countFor($pid);
					$ratings = $this->ProductRatings->averageFor($pid);
					$sold = $this->OrderProducts->countSold($pid);
					$shipping = $this->DeliveryAddresses->defaultFor($user->id);
					$this->Products->touch($pid);
					$details->favourite = (bool)$favs;
					$details->favourite_count = $favCount;
					$details->sold = $sold;
					$details->TotalRatings = ['avg' => $ratings['avg'] ?? 0, 'totalRatings' => $ratings['count'] ?? 0];
					$details->Shipping = $shipping;
					$output = ['code' => 200, 'msg' => $details];
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function showFavouriteProducts(): ResponseInterface
			{
				$this->loadModel('ProductFavourites');
				$this->loadModel('Products');
				$this->loadModel('ProductRatings');
				$this->loadModel('OrderProducts');
				$this->loadModel('DeliveryAddresses');
				$data = $this->getRequest()->getData();
				$user_id = (int)$data['user_id'];
				$start = (int)($data['starting_point'] ?? 0);
				$favs = $this->ProductFavourites->forUser($user_id, $start);
				$output = ['code' => 400, 'msg' => []];
				if ($favs) {
					foreach ($favs as &$p) {
						$pid = $p->product_id;
						$rating = $this->ProductRatings->averageFor($pid);
						$sold = $this->OrderProducts->countSold($pid);
						$p->sold = $sold;
						$p->TotalRatings = ['avg' => $rating['avg'] ?? 0, 'totalRatings' => $rating['count'] ?? 0];
					}
					$output = ['code' => 200, 'msg' => $favs];
				}
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}

			public function postVideoContent(): ResponseInterface
			{
				$this->loadModel('Videos');
				$this->loadModel('Sounds');
				$this->loadModel('Hashtags');
				$this->loadModel('HashtagVideos');
				$this->loadModel('Users');
				$this->loadModel('Notifications');
				$this->loadModel('Followers');
				$this->loadModel('PushNotifications');
				$this->loadModel('VideoProducts');
				$this->loadModel('Stores');
				$this->loadModel('VideoTemps');
				$this->loadModel('Locations');
				$data = $this->getRequest()->getData();
				$user_id = (int)$data['user_id'];
				$temp = $this->VideoTemps->getForUser($user_id, (int)$data['video_temp_id']);
				if (!$temp) {
					return $this->response->withStatus(400);
				}
				$video = [
					'sound_id' => $data['sound_id'],
					'gif' => $temp->gif_url,
					'duration' => $temp->video_duration,
					'video' => $temp->video_url,
					'thum' => $temp->thumb_small_url,
					'thum_small' => $temp->thumb_small_url,
					'description' => $data['description'],
					'privacy_type' => strtolower($data['privacy_type']),
					'allow_comments' => $data['allow_comments'],
					'allow_duet' => $data['allow_duet'],
					'location_name' => $data['location_name'],
					'location_string' => $data['location_string'],
					'lat' => $data['lat'],
					'long' => $data['long'],
					'width' => $temp->video_width,
					'height' => $temp->video_height,
					'user_id' => $user_id,
					'created' => date('Y-m-d H:i:s')
				];
				if ($data['sound_id'] < 1 && $temp->sound_url) {
					$s = ['audio' => $temp->sound_url, 'duration' => $temp->sound_duration, 'thum' => $temp->sound_thumb, 'name' => $temp->sound_name, 'uploaded_by' => 'user'];
					$soundEntity = $this->Sounds->save($this->Sounds->newEntity($s));
					$video['sound_id'] = $soundEntity->id;
				}
				if (!empty($data['products_json'])) {
					$video['tag_product'] = 1;
				}
				$loc = $this->Locations->findByPlaceId($data['google_place_id'])->first();
				if (!$loc) {
					$locData = ['google_place_id' => $data['google_place_id'], 'name' => $data['location_name'], 'lat' => $data['lat'], 'long' => $data['long'], 'location_string' => $data['location_string'], 'created' => date('Y-m-d H:i:s')];
					$loc = $this->Locations->save($this->Locations->newEntity($locData));
				}
				$video['location_id'] = $loc->id;
				$videoEntity = $this->Videos->save($this->Videos->newEntity($video));
				$this->VideoTemps->delete($temp);
				if (!empty($data['products_json'])) {
					foreach (json_decode($data['products_json'], true) as $prod) {
						$store = $this->Stores->findByMealmeId($prod['store_id'])->first();
						$store_id = $store ? $store->id : $prod['store_id'];
						$vp = ['video_id' => $videoEntity->id, 'store_id' => $store_id, 'product_id' => $prod['product_id'], 'product_mealme_json' => json_encode($prod['product_mealme_json'])];
						$this->VideoProducts->save($this->VideoProducts->newEntity($vp));
					}
				}
				if (!empty($data['hashtags_json'])) {
					foreach (json_decode($data['hashtags_json'], true) as $h) {
						$name = strtolower($h['name']);
						$hashtag = $this->Hashtags->findByName($name)->first() ?? $this->Hashtags->save($this->Hashtags->newEntity(['name' => $name]));
						$this->HashtagVideos->save($this->HashtagVideos->newEntity(['video_id' => $videoEntity->id, 'hashtag_id' => $hashtag->id]));
					}
				}
				if (!empty($data['users_json'])) {
					$author = $this->Users->get($user_id);
					foreach (json_decode($data['users_json'], true) as $u) {
						$tagged = $this->Users->get($u['user_id']);
						$msg = $author->username . ' has tagged you in a video';
						$this->Notifications->save($this->Notifications->newEntity(['sender_id' => $user_id, 'receiver_id' => $tagged->id, 'type' => 'video_tag', 'video_id' => $videoEntity->id, 'string' => $msg, 'created' => date('Y-m-d H:i:s')]));
						$pn = $this->PushNotifications->get($tagged->id);
						if ($pn && $pn->video_updates) {
							$this->Firebase->send(['to' => $tagged->device_token, 'notification' => ['title' => $msg, 'body' => '', 'type' => 'video_tag', 'receiver_id' => $tagged->id]]);
						}
					}
				}
				$followers = $this->Followers->forUser($user_id);
				foreach ($followers as $f) {
					$msg = $author->username . ' has posted a video';
					$this->Notifications->save($this->Notifications->newEntity(['sender_id' => $user_id, 'receiver_id' => $f->follower_id, 'type' => 'video_updates', 'video_id' => $videoEntity->id, 'string' => $msg, 'created' => date('Y-m-d H:i:s')]));
					$pn = $this->PushNotifications->get($f->follower_id);
					if ($pn && $pn->video_updates) {
						$this->Firebase->send(['to' => $f->device_token, 'notification' => ['title' => $msg, 'body' => '', 'type' => 'video_new_post', 'receiver_id' => $f->follower_id]]);
					}
				}
				$output = ['code' => 200, 'msg' => $this->Videos->get($videoEntity->id)];
				return $this->response
					->withType('application/json')
					->withStringBody(json_encode($output));
			}
    
			public function clearCache(): void
			{
				$cache = Cache::read('interest_section', 'default');
				debug($cache);
				Cache::clearAll();
			}

			public function addUserLanguage(): void
			{
				$this->loadModel('Language');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$userId = $data['user_id'] ?? null;
					$languages = $data['language'] ?? [];
					$saveData = [];
					foreach ($languages as $lang) {
						$saveData[] = ['lang_id' => $lang['lang_id'], 'user_id' => $userId];
					}
					if ($saveData) {
						$this->Language->saveMany($saveData);
					}
					$userDetails = $this->User->getUserDetailsFromID($userId);
					$output = ['code' => 200, 'msg' => $userDetails];
					$this->response = $this->response
						->withType('application/json')
						->withStringBody(json_encode($output));
				}
			}

			public function addUserInterest(): void
			{
				$this->loadModel('UserInterest');
				$this->loadModel('User');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$user = $this->user_details;
					if (empty($user)) {
						Message::UNAUTHORIZED();
						return;
					}
					$userId = $user['User']['id'];
					$interests = $data['interests'] ?? [];
					$this->UserInterest->deleteAllInterests($userId);
					$saveData = [];
					foreach ($interests as $i) {
						$saveData[] = ['interest_id' => $i['interest_id'], 'user_id' => $userId];
					}
					if ($saveData) {
						$this->UserInterest->saveMany($saveData);
					}
					Cache::delete("user_interests_{$userId}");
					$userDetails = $this->User->getUserDetailsFromID($userId);
					$output = ['code' => 200, 'msg' => $userDetails];
					$this->response = $this->response
						->withType('application/json')
						->withStringBody(json_encode($output));
				}
			}

			public function addUserLanguagePreferences(): void
			{
				$this->loadModel('LanguagePreference');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$userId = $data['user_id'] ?? null;
					$prefs = $data['language'] ?? [];
					$this->LanguagePreference->deleteAllPreference($userId);
					$saveData = [];
					foreach ($prefs as $p) {
						$saveData[] = ['lang_id' => $p['lang_id'], 'user_id' => $userId];
					}
					if ($saveData) {
						$this->LanguagePreference->saveMany($saveData);
					}
					$userDetails = $this->User->getUserDetailsFromID($userId);
					$output = ['code' => 200, 'msg' => $userDetails];
					$this->response = $this->response
						->withType('application/json')
						->withStringBody(json_encode($output));
				}
			}

			public function showLanguages(): void
			{
				$this->loadModel('Language');
				$this->loadModel('LanguagePreference');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$userId = $data['user_id'] ?? null;
					$languages = $this->Language->getAll();
					foreach ($languages as &$lang) {
						$lang['Language']['selected'] = $this->LanguagePreference
							->ifLanguageIsSelected($userId, $lang['Language']['id']) ? 1 : 0;
					}
					$output = ['code' => 200, 'msg' => $languages];
					$this->response = $this->response
						->withType('application/json')
						->withStringBody(json_encode($output));
				}
			}

			public function showUserInterests(): void
			{
				$this->loadModel('Interest');
				$this->loadModel('UserInterest');
				if ($this->request->is('post')) {
					$user = $this->user_details;
					if (empty($user)) {
						Message::UNAUTHORIZED();
						return;
					}
					$userId = $user['User']['id'];
					$details = $this->geUserInterestsFromCache($userId);
					if ($details) {
						$output = ['code' => 200, 'msg' => $details];
					} else {
						Message::EMPTYDATA();
						return;
					}
					$this->response = $this->response
						->withType('application/json')
						->withStringBody(json_encode($output));
				}
			}

			public function watchLiveStream(): void
			{
				$this->loadModel('LiveStreamingWatch');
				$this->loadModel('LiveStreaming');
				$this->loadModel('User');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$uId = $data['user_id'];
					$sId = $data['live_streaming_id'];
					$coin = $data['coin'];
					$sender = $this->User->getUserDetailsFromID($uId);
					$receiver = $this->LiveStreaming->getDetails($sId);
					$watch = $this->LiveStreamingWatch->checkDuplicate($uId, $sId);
					if ($watch && $watch['LiveStreamingWatch']['block'] > 0) {
						$output = ['code' => 201, 'msg' => 'You have been blocked from watching this live stream'];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						return;
					}
					if ($sender) {
						$wallet = $this->walletTotal($uId)['total'];
						if ($wallet < $coin) {
							$output = ['code' => 201, 'msg' => 'You do not have enough coins to send gift'];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
							return;
						}
					}
					$this->LiveStreaming->updateAll(
						['earn_coin' => $receiver['LiveStreaming']['earn_coin'] + $coin],
						['id' => $sId]
					);
					$this->LiveStreamingWatch->save(['user_id' => $uId, 'live_streaming_id' => $sId, 'coin' => $coin]);
					$sender['User']['wallet'] = $this->walletTotal($uId)['total'];
					$output = ['code' => 200, 'msg' => $sender];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function watchVideo(): void
			{
				$this->loadModel('VideoWatch');
				$this->loadModel('Promotion');
				$this->loadModel('Video');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$userId = $data['user_id'];
					$now = date('Y-m-d H:i:s');
					$records = $data['watch_videos'] ?? [];
					$save = [];
					foreach ($records as $r) {
						$save[] = [
							'video_id' => $r['video_id'],
							'user_id' => $userId,
							'duration' => $r['duration'],
							'created' => $now,
							'promotion_id' => $this->Promotion
								->getActivePromotionAgainstVideoID($r['video_id'], $now)['Promotion']['id'] ?? null
						];
						$this->Video->updateAll(
							['view = view + 1'],
							['id' => $r['video_id']]
						);
					}
					if ($save) {
						$this->VideoWatch->saveMany($save);
					}
					$output = ['code' => 200, 'msg' => 'success'];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function purchaseCoin(): void
			{
				$this->loadModel('PurchaseCoin');
				$this->loadModel('User');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$entry = [
						'user_id'        => $data['user_id'],
						'title'          => $data['title'],
						'coin'           => $data['coin'],
						'price'          => $data['price'],
						'transaction_id' => $data['transaction_id'],
						'device'         => $data['device'],
						'created'        => date('Y-m-d H:i:s')
					];
					if ($this->User->exists(['id' => $data['user_id']])) {
						$this->PurchaseCoin->save($entry);
						$detail = $this->PurchaseCoin->getDetails($this->PurchaseCoin->id);
						$detail['User']['wallet'] = $this->walletTotal($data['user_id'])['total'];
						$output = ['code' => 200, 'msg' => $detail];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					}
				}
			}

			public function showCoinWorth(): void
			{
				$this->loadModel('CoinWorth');
				if ($this->request->is('post')) {
					$all = $this->CoinWorth->getAll();
					if ($all) {
						$output = ['code' => 200, 'msg' => $all];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					} else {
						Message::EMPTYDATA();
					}
				}
			}

			public function showGifts(): void
			{
				$this->loadModel('Gift');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$gifts = $data['id']
						? $this->Gift->getDetails($data['id'])
						: (isset($data['type'])
							? $this->Gift->getAllAgainstType($data['type'])
							: $this->Gift->getAll());
					$output = ['code' => 200, 'msg' => $gifts];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function showAppSlider(): void
			{
				$this->loadModel('AppSlider');
				if ($this->request->is('post')) {
					$images = $this->AppSlider->getAll();
					$output = ['code' => 200, 'msg' => $images];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function showCartSlider(): void
			{
				$this->loadModel('CartSlider');
				if ($this->request->is('post')) {
					$images = $this->CartSlider->getAll();
					$output = ['code' => 200, 'msg' => $images];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function showShopSlider(): void
			{
				$this->loadModel('ShopSlider');
				if ($this->request->is('post')) {
					$images = $this->ShopSlider->getAll();
					$output = ['code' => 200, 'msg' => $images];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function showUserActionsData(): void
			{
				$this->loadModel('VideoLike');
				$this->loadModel('VideoFavourite');
				$this->loadModel('RepostVideo');
				$this->loadModel('User');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$user = $this->User->getDetailsAgainstAuthToken($data['auth_token']);
					if (!$user) {
						Message::EMPTYDATA();
						return;
					}
					$userId = $user['User']['id'];
					$output = [
						'code' => 200,
						'msg'  => [
							'video_likes'      => $this->VideoLike->getUserVideoLikes($userId),
							'video_favourites' => $this->VideoFavourite->getUserVideoFavourites($userId),
							'video_reposts'    => $this->RepostVideo->getUserVideoReposts($userId)
						]
					];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function geVideoDetailsFromCache(mixed $videoId): mixed
			{
				$this->loadModel('Video');
				$cacheKey = "video_details_{$videoId}";
				Cache::setConfig('default', ['duration' => '+30 days']);
				$details = Cache::read($cacheKey, 'default');
				if ($details) {
					return $details;
				}
				$details = $this->Video->getObjectsForCache($videoId);
				Cache::write($cacheKey, $details, 'default');
				return $details;
			}

			public function geUserInterestsFromCache(mixed $userId): mixed
			{
				$this->loadModel('UserInterest');
				$cacheKey = "user_interests_{$userId}";
				Cache::setConfig('default', ['duration' => '+30 days']);
				$details = Cache::read($cacheKey, 'default');
				if ($details !== false) {
					return $details;
				}
				$details = $this->UserInterest->getAllAgainstUser($userId);
				Cache::write($cacheKey, $details, 'default');
				return $details;
			}

			public function getUserDetailsFromCache(mixed $userId): mixed
			{
				$this->loadModel('User');
				$cacheKey = "user_details_{$userId}";
				Cache::setConfig('default', ['duration' => '+30 days']);
				$details = Cache::read($cacheKey, 'default');
				if ($details) {
					return $details;
				}
				$details = $this->User->getObjectsForCache($userId);
				Cache::write($cacheKey, $details, 'default');
				return $details;
			}

			public function getProductDetailsFromCache(mixed $id, string $type): mixed
			{
				$this->loadModel('Product');
				$cacheKey = "product_details_{$id}";
				Cache::setConfig('default', ['duration' => '+30 days']);
				$details = Cache::read($cacheKey, 'default');
				if ($details) {
					return $details;
				}
				$details = $this->Product->getObjectsForCache($id, $type);
				if ($details) {
					Cache::write($cacheKey, $details, 'default');
				}
				return $details ?: false;
			}

			public function getOrderDetailsFromCache(mixed $id, string $type): mixed
			{
				$this->loadModel('Store');
				$cacheKey = $type === 'user'
					? "order_details_{$id}"
					: "store_details_{$id}";
				Cache::setConfig('default', ['duration' => '+30 days']);
				$details = Cache::read($cacheKey, 'default');
				if ($details) {
					return $details;
				}
				$model = $type === 'user' ? $this->Order : $this->Store;
				$details = $model->getObjectsForCache($id, $type);
				if ($details) {
					Cache::write($cacheKey, $details, 'default');
				}
				return $details ?: false;
			}

			public function showDiscoverySections(): void
			{
				$this->loadModel('HashtagVideo');
				$this->loadModel('Follower');
				$this->loadModel('VideoProduct');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$userId = $this->user_details['User']['id'] ?? 0;
					$start = $data['starting_point'] ?? 0;
					$cacheKey = $start < 1 ? 'top_hashtags' : null;
					$hashtags = $cacheKey
						? Cache::read($cacheKey, 'default') ?: $this->HashtagVideo->getHastagsWhichHasGreaterNoOfVideos(0)
						: $this->HashtagVideo->getHastagsWhichHasGreaterNoOfVideos($start);
					if ($cacheKey && !$hashtags) {
						Cache::write($cacheKey, $hashtags, 'default');
					}
					$result = [];
					foreach ($hashtags as $h) {
						$videos = Cache::read("hashtag_{$h['Hashtag']['id']}", 'default')
							  ?: $this->HashtagVideo->getHashtagVideosLimit($h['Hashtag']['id']);
						Cache::write("hashtag_{$h['Hashtag']['id']}", $videos, 'default');
						foreach ($videos as &$v) {
							$v['VideoProduct'] = $this->VideoProduct->getOnlyDetailsAgainstVideo($v['Video']['id']) ?: [];
							$this->customizeVideoResponse($v, $userId);
						}
						$result[] = [
							'Hashtag'     => $h['Hashtag'] + ['views' => $h[0]['total_views'], 'videos_count' => $this->HashtagVideo->countHashtagVideos($h['Hashtag']['id'])],
							'Videos'      => $videos
						];
					}
					if ($result) {
						$output = ['code' => 200, 'msg' => $result];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					} else {
						Message::EMPTYDATA();
					}
				}
			}

			public function snsNotification(): void
			{
				$this->loadModel('Video');
				$this->loadModel('Sound');
				if ($this->request->is('post')) {
					$data = json_decode($this->request->input(), true);
					file_put_contents(LOGS . 'sns.txt', print_r($data, true));
					$msg = json_decode($data['Message'], true)['detail'];
					if ($msg['status'] === 'COMPLETE') {
						$video = $this->Video->getVideoDetailsAgainstJobID($msg['jobId']);
						if ($video) {
							// handle video
						} else {
							$sound = $this->Sound->getVideoDetailsAgainstJobID($msg['jobId']);
							if ($sound) {
								// handle sound
							}
						}
					}
				}
			}

			public function showOrderSession(): void
			{
				$this->loadModel('OrderSession');
				if ($this->request->is('post')) {
					$id = $this->request->getData('id');
					$details = $this->OrderSession->getDetails($id);
					$output = ['code' => 200, 'msg' => $details];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				} else {
					Message::EMPTYDATA();
				}
			}

			public function showEarningsHistory(): void
			{
				$this->loadModel('Order');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$details = $this->Order->getListOfOrdersAgainstReferralUserID($data['user_id'], $data['starting_point'] ?? 0);
					$stat = $this->Order->getTotalCommissionEarned($data['user_id'])[0];
					if ($details) {
						$output = [
							'code' => 200,
							'msg'  => $details,
							'stat' => [
								'total_earned' => $stat['total_amount'] / 100,
								'total_orders' => $stat['total_orders']
							]
						];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					} else {
						Message::EMPTYDATA();
					}
				}
			}

			public function showWithdrawalHistory(): void
			{
				$this->loadModel('WithdrawRequest');
				if ($this->request->is('post')) {
					if (empty($this->user_details)) {
						Message::UNAUTHORIZED();
						return;
					}
					$data = $this->request->getData();
					$details = $this->WithdrawRequest->getWithdrawRequestsAgainstUserID($this->user_details['User']['id'], $data['starting_point'] ?? 0);
					if ($details) {
						$output = ['code' => 200, 'msg' => $details];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					} else {
						Message::EMPTYDATA();
					}
				}
			}

			public function postCommentOnVideo(): void
			{
				$this->loadModel('VideoComment');
				$this->loadModel('Notification');
				$this->loadModel('PushNotification');
				if ($this->request->is('post')) {
					$data = $this->request->getData();
					$required = ['video_id', 'comment'];
					foreach ($required as $field) {
						if (empty($data[$field])) {
							Message::BADREQUEST();
							return;
						}
					}
					if (empty($this->user_details)) {
						Message::UNAUTHORIZED();
						return;
					}
					$userId = $this->user_details['User']['id'];
					$commentData = [
						'video_id'  => $data['video_id'],
						'user_id'   => $userId,
						'comment'   => $data['comment'],
						'parent_id' => $data['parent_id'] ?? 0,
						'created'   => date('Y-m-d H:i:s')
					];
					Cache::delete("videocomments_{$data['video_id']}");
					$video = $this->geVideoDetailsFromCache($data['video_id']);
					if (empty($video)) {
						Message::DONOTEXIST('Video');
						return;
					}
					$this->VideoComment->save($commentData);
					$detail = $this->VideoComment->getDetails($this->VideoComment->id);
					// notifications omitted for brevity
					$output = ['code' => 200, 'msg' => $detail];
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function downloadVideo(): void
			{
				$this->loadModel('Video');
				if ($this->request->is('post')) {
					$videoId = $this->request->getData('video_id');
					$video = $this->Video->getOnlyVideoDetails($videoId);
					if (empty($video)) {
						Message::DONOTEXIST('Video');
						return;
					}
					$ffmpeg = new Ffmpeg();
					$url = $video['Video']['video_with_watermark'] ?? null;
					if ($url && strpos($url, 'invalid') === false) {
						$output = ['code' => 200, 'msg' => $url];
					} else {
						$downloaded = $ffmpeg->addWaterMarkAndText($video['Video']['video'], 1, $video['User']['username'], $video['Video']['duration']);
						if (defined('IAM_KEY') && strlen(IAM_KEY) > 5) {
							$res = $this->aws->fileUploadToS3($downloaded, 'mp4');
							$videoUrl = $res['code'] === 200 ? Utility::getCloudFrontUrl($res['msg'], '/video') : $downloaded;
						} else {
							$videoUrl = $downloaded;
						}
						$this->Video->updateAll(['video_with_watermark' => $videoUrl], ['id' => $videoId]);
						$output = ['code' => 200, 'msg' => $videoUrl];
					}
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function deleteWaterMarkVideo(): void
			{
				if ($this->request->is('post')) {
					$url = $this->request->getData('video_url');
					if ($url) {
						@unlink($url);
						$output = ['code' => 200, 'msg' => 'deleted'];
					} else {
						$output = ['code' => 201, 'msg' => 'invalid url'];
					}
					$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
				}
			}

			public function showHtmlPage(): void
			{
				$this->loadModel('HtmlPage');
				$this->autoRender = true;
				$page = $this->request->getQuery('page');
				$exist = $this->HtmlPage->ifExist($page);
				if ($exist) {
					$this->set('data', $exist);
				}
			}
    
				protected function verifyParameters(array $data, array $requiredParams): void
				{
					$missing = [];
					foreach ($requiredParams as $param) {
						if (!array_key_exists($param, $data)) {
							$missing[] = $param;
						}
					}
					if ($missing) {
						throw new BadRequestException('Missing parameters: ' . implode(', ', $missing));
					}
				}

				public function likeVideo(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$this->verifyParameters($data, ['video_id']);
					$videoId = $data['video_id'];
					$user = $this->Authentication->getIdentity();
					if (!$user) {
						throw new UnauthorizedException();
					}
					$userId = $user->getIdentifier();
					$videoDetails = $this->geVideoDetailsFromCache($videoId);
					if (empty($videoDetails)) {
						throw new NotFoundException('Video not found');
					}
					$now = date('Y-m-d H:i:s');
					$likeData = ['video_id' => $videoId, 'user_id' => $userId, 'created' => $now];
					$existing = $this->VideoLike->ifExist($likeData);
					if ($existing) {
						$this->VideoLike->delete($existing['VideoLike']['id']);
						$msg = 'unlike';
						$flag = 0;
					} else {
						$entity = $this->VideoLike->newEntity($likeData);
						$this->VideoLike->save($entity);
						$details = $this->VideoLike->getDetails($entity->id);
						$msg = $details;
						$flag = 1;
						if (empty($details)) {
							throw new NotFoundException('No data');
						}
						if ($userId !== $details['Video']['user_id']) {
							$notificationMsg = $user->username . ' liked your video';
							$payload = [
								'to' => $details['Video']['User']['device_token'],
								'notification' => [
									'title' => $user->username,
									'image' => Router::url('/', true) . $user->profile_pic,
									'body' => $notificationMsg,
									'badge' => '1',
									'sound' => 'default',
									'type' => 'video_like',
									'video_id' => $videoId,
									'receiver_id' => $details['Video']['user_id'],
								],
								'data' => [
									'title' => $user->username,
									'body' => $notificationMsg,
									'icon' => Router::url('/', true) . $user->profile_pic,
									'badge' => '1',
									'sound' => 'default',
									'type' => 'video_like',
									'video_id' => $videoId,
									'receiver_id' => $details['Video']['user_id'],
								]
							];
							$prefs = $this->PushNotification->getDetails($userId);
							if (!empty($prefs) && $prefs['PushNotification']['likes'] > 0) {
								$this->firebase->sendPushNotification($payload);
							}
							$notif = $this->Notification->newEntity([
								'video_id' => $videoId,
								'sender_id' => $userId,
								'receiver_id' => $details['Video']['user_id'],
								'type' => 'video_like',
								'string' => $notificationMsg,
								'created' => $now
							]);
							$this->Notification->save($notif);
							Cache::delete('notifications_' . $details['Video']['user_id']);
						}
					}
					$count = $this->VideoLike->countLikes($videoId);
					$output = ['code' => 200, 'msg' => $msg, 'like' => $flag, 'like_count' => $count];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addVideoFavourite(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$videoId = $data['video_id'];
					$user = $this->Authentication->getIdentity();
					if (!$user) {
						throw new UnauthorizedException();
					}
					$userId = $user->getIdentifier();
					$now = date('Y-m-d H:i:s');
					$favData = ['video_id' => $videoId, 'user_id' => $userId, 'created' => $now];
					$existing = $this->VideoFavourite->ifExist($favData);
					if ($existing) {
						$this->VideoFavourite->delete($existing['VideoFavourite']['id']);
						$msg = 'unfavourite';
						$code = 201;
					} else {
						$entity = $this->VideoFavourite->newEntity($favData);
						$this->VideoFavourite->save($entity);
						$msg = $this->VideoFavourite->getDetails($entity->id);
						$code = 200;
					}
					$count = $this->VideoFavourite->getFavVideosCount($videoId);
					$output = ['code' => $code, 'msg' => $msg, 'fav_count' => $count];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function likeComment(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$this->verifyParameters($data, ['comment_id']);
					$user = $this->Authentication->getIdentity();
					if (!$user) {
						throw new UnauthorizedException();
					}
					$userId = $user->getIdentifier();
					$commentDetails = $this->VideoComment->getDetails($data['comment_id']);
					if (empty($commentDetails)) {
						throw new NotFoundException('Comment not found');
					}
					$now = date('Y-m-d H:i:s');
					$likeData = ['comment_id' => $data['comment_id'], 'user_id' => $userId, 'created' => $now];
					$existing = $this->VideoCommentLike->ifExist($likeData);
					if ($existing) {
						$this->VideoCommentLike->delete($existing['VideoCommentLike']['id']);
						$msg = 'unfavourite';
						$code = 201;
					} else {
						$entity = $this->VideoCommentLike->newEntity($likeData);
						$this->VideoCommentLike->save($entity);
						$details = $this->VideoCommentLike->getDetails($entity->id);
						$details['VideoCommentLike']['owner_like'] = ($details['VideoComment']['Video']['user_id'] === $userId) ? 1 : 0;
						$msg = $details;
						$code = 200;
					}
					$count = $this->VideoCommentLike->countLikes($data['comment_id']);
					$output = ['code' => $code, 'msg' => $msg, 'like_count' => $count];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showUserOrders(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$user = $this->Authentication->getIdentity();
					if (!$user) {
						throw new UnauthorizedException();
					}
					$userId = $user->getIdentifier();
					$data = $this->getRequest()->getData();
					$start = $data['starting_point'] ?? 0;
					$orders = $this->Order->getUserOrders($userId, 0, $start);
					if (empty($orders)) {
						throw new NotFoundException('No orders');
					}
					$output = ['code' => 200, 'msg' => $orders];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showRooms(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$details = isset($data['club_id'])
						? $this->Room->getRoomsAgainstClub($data['club_id'])
						: $this->Room->getAll();
					if (empty($details)) {
						throw new NotFoundException('No rooms');
					}
					$output = ['code' => 200, 'msg' => $details];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showOrderDetail(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$order = $this->Order->getDetails($data['order_id']);
					if (empty($order)) {
						throw new NotFoundException('Order not found');
					}
					$output = ['code' => 200, 'msg' => $order];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showStoreOrders(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$storeId = $data['store_id'];
					$orders = $this->Order->getStoreOrders($storeId);
					if (empty($orders)) {
						throw new NotFoundException('No orders');
					}
					$output = ['code' => 200, 'msg' => $orders];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addProductAttribute(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$now = date('Y-m-d H:i:s');
					$payload = [
						'product_id' => $data['product_id'],
						'name' => $data['name'],
						'created' => $now
					];
					if (isset($data['id'])) {
						$entity = $this->ProductAttribute->get($data['id']);
						$this->ProductAttribute->patchEntity($entity, $payload);
						$this->ProductAttribute->save($entity);
						$msg = $this->ProductAttribute->getDetails($data['id']);
					} else {
						$existing = $this->ProductAttribute->ifExist($payload);
						if ($existing) {
							throw new BadRequestException('Duplicate');
						}
						$entity = $this->ProductAttribute->newEntity($payload);
						$this->ProductAttribute->save($entity);
						$msg = $this->ProductAttribute->getDetails($entity->id);
					}
					$output = ['code' => 200, 'msg' => $msg];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showDailyCheckins(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$userDetails = $this->User->getOnlyUserDetailsFromID($data['user_id']);
					if (empty($userDetails)) {
						throw new NotFoundException('User not found');
					}
					$now = date('Y-m-d H:i:s');
					$regDate = new \DateTime($userDetails['User']['created']);
					$today = new \DateTime($now);
					$startDay = $regDate->format('l');
					$todayDay = $today->format('l');
					$newDate = $todayDay === $startDay
						? $today->format('Y-m-d')
						: $this->getRecentPastDate($startDay);
					$checkins = $this->DailyCheckin->getRecentCheckins($data['user_id'], $newDate);
					$output = [
						'code' => 200,
						'msg' => ['checkins' => $checkins, 'starting_date' => $newDate, 'server_datetime' => $now]
					];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addDocument(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					if (!empty($data['auth_token'])) {
						$user = $this->User->getDetailsAgainstAuthToken($data['auth_token']);
						if (empty($user)) {
							throw new UnauthorizedException();
						}
						$userId = $user['User']['id'];
					}
					$type = $data['type'];
					$documents = $data['documents'];
					$now = date('Y-m-d H:i:s');
					$entities = [];
					foreach ($documents as $doc) {
						$entities[] = $this->Document->newEntity([
							'title' => $doc['title'],
							'type' => $type,
							'user_id' => $userId,
							'value' => $doc['value'],
							'created' => $now
						]);
					}
					$this->Document->saveMany($entities);
					$output = ['code' => 200, 'msg' => 'success'];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addDailyCheckin(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$userDetails = $this->User->getOnlyUserDetailsFromID($data['user_id']);
					if (empty($userDetails)) {
						throw new NotFoundException('User not found');
					}
					$now = date('Y-m-d H:i:s');
					$today = date('Y-m-d');
					$ifExist = $this->DailyCheckin->ifExist($data['user_id'], $today);
					if ($ifExist) {
						$output = ['code' => 201, 'msg' => 'Already exist'];
					} else {
						$entity = $this->DailyCheckin->newEntity([
							'user_id' => $data['user_id'],
							'coin' => $data['coin'],
							'created' => $now
						]);
						$this->DailyCheckin->save($entity);
						$details = $this->DailyCheckin->getDetails($entity->id);
						$output = ['code' => 200, 'msg' => $details];
					}
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function getRecentPastDate(string $day): string
				{
					$now = new \DateTime();
					$past = (clone $now)->modify("last $day");
					return $past->format('Y-m-d');
				}

				public function addCategoryInProduct(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$details = $this->Product->getDetails($data['product_id']);
					if (empty($details)) {
						throw new NotFoundException('Product not found');
					}
					$this->Product->patchEntity($details, ['product_category_id' => $data['product_category_id']]);
					$this->Product->save($details);
					$msg = $this->Product->getDetails($data['product_id']);
					$output = ['code' => 200, 'msg' => $msg];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addProductAttributeVariation(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$now = date('Y-m-d H:i:s');
					if (!empty($data['option'])) {
						$this->ProductAttributeVariation->deleteAllVariations($data['product_attribute_id']);
						$entities = [];
						foreach ($data['option'] as $opt) {
							$entities[] = $this->ProductAttributeVariation->newEntity([
								'product_attribute_id' => $data['product_attribute_id'],
								'value' => $opt['value'],
								'created' => $now
							]);
						}
						$this->ProductAttributeVariation->saveMany($entities);
					}
					$details = $this->ProductAttribute->getDetails($data['product_attribute_id']);
					$output = ['code' => 200, 'msg' => $details];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addProductAttributeCombination(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$now = date('Y-m-d H:i:s');
					if (!empty($data['option'])) {
						$this->ProductAttributeCombination->deleteAllCombinations($data['product_id']);
						$entities = [];
						foreach ($data['option'] as $opt) {
							$entities[] = $this->ProductAttributeCombination->newEntity([
								'product_id' => $data['product_id'],
								'combination_string' => $opt['combination_string'],
								'sku' => $opt['sku'],
								'price' => $opt['price'],
								'available_stock' => $opt['available_stock'],
								'created' => $now
							]);
						}
						$this->ProductAttributeCombination->saveMany($entities);
					}
					$output = ['code' => 200, 'msg' => 'success'];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addHashtagFavourite(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$user = $this->Authentication->getIdentity();
					if (!$user) {
						throw new UnauthorizedException();
					}
					$userId = $user->getIdentifier();
					$now = date('Y-m-d H:i:s');
					$favData = ['hashtag_id' => $data['hashtag_id'], 'user_id' => $userId, 'created' => $now];
					$existing = $this->HashtagFavourite->ifExist($favData);
					if ($existing) {
						$this->HashtagFavourite->delete($existing['HashtagFavourite']['id']);
						$msg = 'unfavourite';
						$code = 201;
					} else {
						$entity = $this->HashtagFavourite->newEntity($favData);
						$this->HashtagFavourite->save($entity);
						$msg = $this->HashtagFavourite->getDetails($entity->id);
						$code = 200;
					}
					$output = ['code' => $code, 'msg' => $msg];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showFavouriteHashtags(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$start = $data['starting_point'] ?? 0;
					$fav = $this->HashtagFavourite->getUserFavouriteHashtags($data['user_id'], $start);
					if (empty($fav)) {
						throw new NotFoundException('No data');
					}
					foreach ($fav as $i => $h) {
						$views = $this->HashtagVideo->countHashtagViews($h['Hashtag']['id']);
						$count = $this->HashtagVideo->countHashtagVideos($h['Hashtag']['id']);
						$fav[$i]['Hashtag']['views'] = $views[0]['total_sum'] ?? 0;
						$fav[$i]['Hashtag']['videos_count'] = $count;
					}
					$output = ['code' => 200, 'msg' => $fav];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function addSoundFavourite(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$user = $this->Authentication->getIdentity();
					if (!$user) {
						throw new UnauthorizedException();
					}
					$userId = $user->getIdentifier();
					$now = date('Y-m-d H:i:s');
					$favData = ['sound_id' => $data['sound_id'], 'user_id' => $userId, 'created' => $now];
					$existing = $this->SoundFavourite->ifExist($favData);
					if ($existing) {
						$this->SoundFavourite->delete($existing['SoundFavourite']['id']);
						$msg = 'unfavourite';
					} else {
						$entity = $this->SoundFavourite->newEntity($favData);
						$this->SoundFavourite->save($entity);
						$msg = $this->SoundFavourite->getDetails($entity->id);
					}
					$output = ['code' => 200, 'msg' => $msg];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showUsersAccordingToGiftReceive(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$type = $data['type'];
					$start = $data['starting_point'] ?? 0;
					$date = date('Y-m-d');
					if ($type === 'daily') {
						$senders = $this->GiftSend->getDailyReceivers($date, $start);
					} else {
						$hour = (new \DateTime())->modify('-1 hour')->format('H');
						$senders = $this->GiftSend->getHourlyReceivers($hour, $start);
					}
					if (empty($senders)) {
						throw new NotFoundException('No data');
					}
					foreach ($senders as $i => $s) {
						$sum = $this->GiftSend->countGiftReceiveByUser($s['GiftSend']['sender_id']);
						$senders[$i]['GiftSend']['total_coins'] = $sum[0]['total_amount'] ?? '0';
					}
					$output = ['code' => 200, 'msg' => ['data' => $senders, 'server_datetime' => date('Y-m-d H:i:s')]];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showUsersAccordingToGiftSent(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$type = $data['type'];
					$start = $data['starting_point'] ?? 0;
					$date = date('Y-m-d');
					if ($type === 'daily') {
						$senders = $this->GiftSend->getDailySenders($date, $start);
					} else {
						$hour = (new \DateTime())->modify('-1 hour')->format('H');
						$senders = $this->GiftSend->getHourlySenders($hour, $start);
					}
					if (empty($senders)) {
						throw new NotFoundException('No data');
					}
					foreach ($senders as $i => $s) {
						$sum = $this->GiftSend->countGiftSendByUser($s['GiftSend']['sender_id']);
						$top = $this->GiftSend->getTopGiftsSendByUser($s['GiftSend']['sender_id']);
						$senders[$i]['GiftSend']['total_coins'] = $sum[0]['total_amount'] ?? '0';
						$senders[$i]['User']['top_gifts'] = $top;
					}
					$output = ['code' => 200, 'msg' => ['data' => $senders, 'server_datetime' => date('Y-m-d H:i:s')]];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showFavouriteSounds(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$start = $data['starting_point'] ?? 0;
					$fav = $this->SoundFavourite->getUserFavouriteSounds($data['user_id'], $start);
					if (empty($fav)) {
						throw new NotFoundException('No data');
					}
					$output = ['code' => 200, 'msg' => $fav];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showPromotions(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$details = $this->Promotion->getUserPromotions(
						$data['user_id'],
						$data['start_datetime'],
						$data['end_datetime'],
						$data['starting_point']
					);
					if (empty($details)) {
						throw new NotFoundException('No promotions');
					}
					$stats = ['total_coins' => 0, 'total_destination_tap' => 0, 'total_likes' => 0, 'total_views' => 0];
					foreach ($details as $i => $d) {
						$views = $this->VideoWatch->countWatchVideos([$d['Promotion']['video_id']], $d['Promotion']['start_datetime'], $d['Promotion']['end_datetime']);
						$likes = $this->VideoLike->countLikesBetweenDatetime([$d['Promotion']['video_id']], $d['Promotion']['start_datetime'], $d['Promotion']['end_datetime']);
						$details[$i]['Promotion']['video_views'] = $views;
						$stats['total_coins'] += $d['Promotion']['coin'];
						$stats['total_destination_tap'] += $d['Promotion']['destination_tap'];
						$stats['total_likes'] += $likes;
						$stats['total_views'] += $views;
					}
					$output = ['code' => 200, 'msg' => ['Details' => $details, 'Stats' => $stats]];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function useSticker(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$sticker = $this->Sticker->get($data['sticker_id']);
					if (empty($sticker)) {
						throw new NotFoundException('Sticker not found');
					}
					$this->StickerUsed->save($this->StickerUsed->newEntity([
						'user_id' => $data['user_id'],
						'sticker_id' => $data['sticker_id']
					]));
					$this->Sticker->patchEntity($sticker, ['used_count' => $sticker->used_count + 1]);
					$this->Sticker->save($sticker);
					$output = ['code' => 200, 'msg' => 'success'];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showSettings(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$settings = isset($data['id'])
						? $this->Setting->getDetails($data['id'])
						: $this->Setting->getAll();
					if (empty($settings)) {
						throw new NotFoundException('No settings');
					}
					$output = ['code' => 200, 'msg' => $settings];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}

				public function showStickers(): ResponseInterface
				{
					if (!$this->getRequest()->is('post')) {
						throw new BadRequestException();
					}
					$data = $this->getRequest()->getData();
					$fav = $this->Sticker->getAll($data['type'], $data['starting_point']);
					if (empty($fav)) {
						throw new NotFoundException('No stickers');
					}
					$output = ['code' => 200, 'msg' => $fav];
					return $this->response
						->withType('application/json')
						->withStringBody((string)json_encode($output));
				}
    
					public function showFavouriteVideos(): void
					{
						$this->loadModel('VideoFavourite');
						$this->loadModel('VideoLike');
						$this->loadModel('Follower');
						$this->loadModel('VideoComment');
						$this->loadModel('RepostVideo');
						$this->loadModel('VideoProduct');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$user_id = 0;
							$user_details = $this->user_details;
							if (!empty($user_details)) {
								$user_id = (int)$user_details['User']['id'];
							}
							$starting_point = isset($data['starting_point']) ? (int)$data['starting_point'] : 0;
							$fav_posts = $this->VideoFavourite->getUserFavouriteVideos($user_id, $starting_point);

							if (!empty($fav_posts)) {
								foreach ($fav_posts as $key => $videos) {
									$video_data = [
										'user_id' => $user_id,
										'video_id' => (int)$videos['Video']['id']
									];
									$fav_posts[$key]['Video']['like'] = $this->VideoLike->ifExist($video_data) ? 1 : 0;
									$fav_posts[$key]['Video']['favourite'] = $this->VideoFavourite->ifExist($video_data) ? 1 : 0;
									$fav_posts[$key]['Video']['repost'] = $this->RepostVideo->ifExist($video_data) ? 1 : 0;
									$fav_posts[$key]['Video']['comment_count']   = $this->VideoComment->countComments((int)$videos['Video']['id']);
									$fav_posts[$key]['Video']['like_count']      = $this->VideoLike->countLikes((int)$videos['Video']['id']);
									$fav_posts[$key]['Video']['favourite_count'] = $this->VideoFavourite->getFavVideosCount((int)$videos['Video']['id']);
									$video_cache = $this->geVideoDetailsFromCache((int)$videos['Video']['id']);
									$user_cache  = $this->getUserDetailsFromCache((int)$videos['Video']['user_id']);
									$fav_posts[$key]['User']   = $user_cache['User'];
									$fav_posts[$key]['User']['PrivacySetting'] = $user_cache['PrivacySetting'];
									$fav_posts[$key]['Sound']   = $video_cache['Sound'];
									$fav_posts[$key]['Location'] = $video_cache['Location'];
									$product = $this->VideoProduct->getOnlyDetailsAgainstVideo((int)$videos['Video']['id']);
									if (!empty($product)) {
										if ((int)$product['VideoProduct']['product_id'] > 0) {
											$pd = $this->getProductDetailsFromCache((int)$product['VideoProduct']['product_id'], 'id');
											if ($pd) {
												$fav_posts[$key]['Product'] = [
													'id'          => (string)$pd['Product']['id'],
													'title'       => (string)$pd['Product']['title'],
													'third_party' => '0'
												];
											}
										} elseif (strlen($product['VideoProduct']['product_id']) > 15 && Utility::isValidJson($product['VideoProduct']['product_mealme_json'])) {
											$mj = json_decode($product['VideoProduct']['product_mealme_json'], true);
											$fav_posts[$key]['Product'] = [
												'id'          => (string)$product['VideoProduct']['product_id'],
												'title'       => (string)$mj['name'],
												'third_party' => '1'
											];
										}
									}
								}

								$output = ['code' => 200, 'msg' => $fav_posts];
								$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
								return;
							}

							Message::EMPTYDATA();
							return;
						}
					}

					public function videoCompressorUpdate(): void
					{
						$this->loadModel('Video');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							file_put_contents(WWW_ROOT . 'test.txt', json_encode($data));
							if (isset($data['code']) && $data['code'] === 200) {
								$video_id = (int)$data['file_id'];
								$url_from_deepengin = $data['compress_url'];
								$video = $this->Video->getOnlyVideoDetails($video_id);
								if (!empty($video)) {
									$local = Utility::fileDownloadFromUrl($url_from_deepengin, UPLOADS_FOLDER_URI, 'mp4');
									if (MEDIA_STORAGE === 's3') {
										$s3 = $this->aws->fileUploadToS3($local, 'mp4', UPLOADS_FOLDER_URI);
										$new = $s3['code'] === 200
											 ? Utility::getCloudFrontUrl($s3['msg'], '/video')
											 : $video['Video']['video'];
										unlink($local);
									} else {
										$new = $local;
									}
									$this->Video->id   = $video_id;
									$this->Video->save(['video' => $new, 'compression' => 1]);
									$this->Video->clear();
									$oldUrl = $video['Video']['video'];
									if (strpos($oldUrl, 'http') !== false) {
										$this->aws->deleteObjectS3($oldUrl);
									} else {
										unlink($oldUrl);
									}
								}
							}
						}
					}

					public function getVideoDetection(): void
					{
						$this->loadModel('Video');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$video_id = (int)$data['file_id'];
							$mod = $data['ModerationResult'];
							$post = ['quality_check' => 1];
							$video = $this->Video->getOnlyVideoDetails($video_id);
							if (!empty($video) && !empty($mod)) {
								$label  = strtolower($mod['Name'] ?? '');
								$parent = strtolower($mod['ParentName'] ?? '');
								if (str_contains($label, 'nudity') || str_contains($parent, 'nudity')) {
									$post['nudity_found'] = 1;
								}
								$this->Video->id = $video_id;
								$this->Video->save($post);
								$this->Video->clear();
							}
						}
					}

					public function liveStream(): void
					{
						$this->loadModel('LiveStreaming');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$stream = [];
							if (isset($data['user_id']))    $stream['user_id']    = (int)$data['user_id'];
							if (isset($data['duration']))   $stream['duration']   = (int)$data['duration'];
							if (isset($data['started_at'])) {
								$stream['started_at'] = $data['started_at'];
							} elseif (isset($data['ended_at'])) {
								$details = $this->LiveStreaming->getDetails((int)$data['id']);
								if (!empty($details)) {
									$stream['ended_at'] = $data['ended_at'];
									$stream['duration'] = Utility::getSeconds($details['LiveStreaming']['started_at'], $data['ended_at']);
								}
							}
							if (isset($data['id'])) {
								$this->LiveStreaming->id = (int)$data['id'];
								$this->LiveStreaming->save($stream);
								$details = $this->LiveStreaming->getDetails((int)$data['id']);
								$output = ['code' => 200, 'msg' => $details];
							} else {
								$stream['created'] = date('Y-m-d H:i:s');
								$this->LiveStreaming->save($stream);
								$details = $this->LiveStreaming->getDetails($this->LiveStreaming->id);
								$output = ['code' => 200, 'msg' => $details];
							}
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
							return;
						}
					}

					public function showUnReadNotifications(): void
					{
						$this->loadModel('Notification');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$user_id = (int)($data['user_id'] ?? 0);
							$notes = $this->Notification->getUserUnreadNotification($user_id);
							if (!empty($notes)) {
								$output = ['code' => 200, 'msg' => $notes];
								$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
								return;
							}
							Message::EMPTYDATA();
							return;
						}
					}

					public function readNotification(): void
					{
						$this->loadModel('Notification');
						$this->loadModel('User');

						if ($this->getRequest()->is('post')) {
							$user = $this->user_details;
							if (empty($user)) {
								Message::UNAUTHORIZED();
								return;
							}
							$user_id = (int)$user['User']['id'];
							$details = $this->Notification->getUserNotifications($user_id, 0);
							if (!empty($details)) {
								$this->Notification->readNotification($user_id);
								$output = ['code' => 200, 'msg' => 'success'];
							} else {
								$output = ['code' => 201, 'msg' => 'no notification exist'];
							}
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
							return;
						}
					}

					public function acceptStreamingInvite(): void
					{
						$this->loadModel('Notification');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$id     = (int)$data['id'];
							$status = $data['status'];
							$details = $this->Notification->getDetails($id);
							if (!empty($details)) {
								$this->Notification->id = $id;
								$this->Notification->saveField('status', $status);
								$updated = $this->Notification->getDetails($id);
								$output  = ['code' => 200, 'msg' => $updated];
								$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
								return;
							}
						}
					}

					public function showUsers(): void
					{
						$this->loadModel('User');
						$this->loadModel('Follower');
						$this->loadModel('Video');

						if ($this->getRequest()->is('post')) {
							$data    = $this->getRequest()->getData();
							$users   = $data['users'] ?? [];
							$user_id = (int)($data['user_id'] ?? 0);
							if (!empty($users)) {
								$ids = array_map(fn($u) => (int)$u['user_id'], $users);
								$details = $this->User->getMultipleUsersData($ids);
								if (!empty($details)) {
									foreach ($details as $key => $u) {
										$f1 = $this->Follower->ifFollowing($user_id, (int)$u['User']['id']);
										$f2 = $this->Follower->ifFollowing((int)$u['User']['id'], $user_id);
										$details[$key]['User']['followers_count']  = $this->Follower->countFollowers((int)$u['User']['id']);
										$details[$key]['User']['following_count'] = $this->Follower->countFollowing((int)$u['User']['id']);
										$details[$key]['User']['video_count']     = $this->Video->getUserVideosCount((int)$u['User']['id']);
										$details[$key]['User']['button']          = $this->getUserRelationship($f1, $f2);
										$details[$key]['User']['notification']    = $f1 ? $f1['Follower']['notification'] : 0;
										$details[$key]['User']['block']           = ($f1 && $f2) ? 1 : 0;
									}
								}
								$output = ['code' => 200, 'msg' => $details];
								$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
								return;
							}
							Message::EMPTYDATA();
							return;
						}
					}

					public function inviteUserToStreaming(): void
					{
						$this->loadModel('Notification');
						$this->loadModel('LiveStreaming');
						$this->loadModel('User');

						if ($this->getRequest()->is('post')) {
							$data     = $this->getRequest()->getData();
							$created  = date('Y-m-d H:i:s');
							$streamId = (int)$data['live_streaming_id'];
							$users    = $data['users'] ?? [];
							$type     = $data['type'] ?? '';

							foreach ($users as $u) {
								$uid      = (int)$u['user_id'];
								$details  = $this->LiveStreaming->getDetails($streamId);
								$receiver = $this->User->getUserDetailsFromID($uid);
								if (!empty($details) && !empty($receiver)) {
									$msg = 'You have been invited in live streaming session';
									$exists = $this->Notification->ifStreamingNotificationExist((int)$details['LiveStreaming']['user_id'], $uid, $streamId);
									if ($exists < 1) {
										$this->Notification->save([
											'sender_id'         => (int)$details['LiveStreaming']['user_id'],
											'receiver_id'       => $uid,
											'live_streaming_id' => $streamId,
											'type'              => $type,
											'string'            => $msg,
											'created'           => $created
										]);
									}
									$notification = [
										'to'           => $receiver['User']['device_token'],
										'notification' => [
											'title' => 'streaming invitation',
											'body'  => $msg,
											'badge' => '1',
											'sound' => 'default',
											'type'  => 'live',
											'live_streaming_id' => $streamId,
											'user_id' => $uid,
											'receiver_id' => $receiver['User']['id']
										],
										'data' => [
											'title' => 'streaming invitation',
											'body'  => $msg,
											'badge' => '1',
											'sound' => 'default',
											'type'  => 'live',
											'live_streaming_id' => $streamId,
											'user_id' => $uid,
											'receiver_id' => $receiver['User']['id']
										]
									];
									$this->firebase->sendPushNotification($notification);
								}
							}

							$output = ['code' => 200, 'msg' => 'success'];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
							return;
						}
					}

					private function populateUserStats(array &$userDetail, int $user_id, int $other_user_id): void
					{
						if ($user_id === $other_user_id) {
							$this->populateCurrentUserStats($userDetail, $user_id);
							return;
						}
						$this->loadModel('Follower');
						$this->loadModel('VideoLike');
						$this->loadModel('Video');
						$this->loadModel('Order');
						$this->loadModel('VideoProduct');
						$this->loadModel('Playlist');

						$created      = date('Y-m-d H:i:s');
						$one_day_ago  = date('Y-m-d H:i:s', strtotime('-1 day', strtotime($created)));

						$userDetail['User']['likes_count']       = $this->VideoLike->countLikesOnAllUserVideos($other_user_id);
						$userDetail['User']['followers_count']   = $this->Follower->countFollowers($other_user_id);
						$userDetail['User']['following_count']   = $this->Follower->countFollowing($other_user_id);
						$userDetail['User']['sold_items_count']  = $this->Order->getCountSoldItems($other_user_id);
						$userDetail['User']['story']             = $this->Video->getUserStory($other_user_id, $one_day_ago);
						$userDetail['User']['video_count']       = $this->Video->getUserVideosCount($other_user_id);
						$userDetail['User']['Playlist']          = $this->Playlist->getUserOnlyPlaylist($other_user_id);

						$f1 = $this->Follower->ifFollowing($user_id, $other_user_id);
						$f2 = $this->Follower->ifFollowing($other_user_id, $user_id);

						$userDetail['User']['button'] = $this->getUserRelationship($f1, $f2);
						$userDetail['User']['block']  = $this->getBlockStatus($user_id, $other_user_id);
					}

					private function populateCurrentUserStats(array &$userDetail, int $user_id): void
					{
						$this->loadModel('Follower');
						$this->loadModel('VideoLike');
						$this->loadModel('Video');
						$this->loadModel('Order');
						$this->loadModel('VideoProduct');
						$this->loadModel('Playlist');
						$this->loadModel('ProfileVisit');

						$created     = date('Y-m-d H:i:s');
						$one_day_ago = date('Y-m-d H:i:s', strtotime('-1 day', strtotime($created)));

						$userDetail['User']['Interests']             = $this->geUserInterestsFromCache($user_id);
						$wallet = $this->walletTotal($user_id);
						$userDetail['User']['wallet']                = $wallet['total'];
						$userDetail['User']['comission_earned']      = $wallet['comission_earned'];
						$userDetail['User']['total_balance_usd']     = $wallet['total_balance_usd'];
						$userDetail['User']['total_all_time_coins']  = $wallet['gifts_receive'];
						$userDetail['User']['story']                 = $this->Video->getUserStory($user_id, $one_day_ago);
						$userDetail['User']['Playlist']              = $this->Playlist->getUserOnlyPlaylist($user_id);

						$userDetail['User']['followers_count']       = $this->Follower->countFollowers($user_id);
						$userDetail['User']['following_count']       = $this->Follower->countFollowing($user_id);
						$userDetail['User']['likes_count']           = $this->VideoLike->countLikesOnAllUserVideos($user_id);
						$userDetail['User']['video_count']           = $this->Video->getUserVideosCount($user_id);
						$userDetail['User']['profile_visit_count']   = $this->ProfileVisit->getProfileVisitorsUnreadCount($user_id) ?? 0;
						$userDetail['User']['sold_items_count']      = $this->Order->getCountSoldItems($user_id);
						$userDetail['User']['tagged_products_count'] = 0;
					}

					private function getUserRelationship(array $follower_details, array $following_details): string
					{
						if (!empty($follower_details) && !empty($following_details)) {
							return 'Friends';
						}
						if (!empty($follower_details)) {
							return 'following';
						}
						if (!empty($following_details)) {
							return 'follow back';
						}
						return 'follow';
					}

					private function getBlockStatus(int $user_id, int $other_user_id): int
					{
						$this->loadModel('BlockUser');
						$b1 = $this->BlockUser->ifBlocked($user_id, $other_user_id);
						$b2 = $this->BlockUser->ifBlocked($other_user_id, $user_id);
						return (!empty($b1) || !empty($b2)) ? 1 : 0;
					}

					public function showUserDetail(): void
					{
						$this->loadModel('Notification');
						$this->loadModel('ProfileVisit');

						if (!$this->getRequest()->is('post')) {
							return;
						}

						$data = $this->getRequest()->getData();
						$user = $this->user_details;
						$user_id = !empty($user) ? (int)$user['User']['id'] : 0;
						$other_user_id = (int)($data['other_user_id'] ?? ($data['user_id'] ?? 0));

						if ($other_user_id > 0) {
							$detail = $this->getUserDetailsFromCache($other_user_id) ?: [];
							if (empty($detail)) {
								Message::EMPTYDATA();
								return;
							}
							$this->populateUserStats($detail, $user_id, $other_user_id);
							if ($other_user_id !== $user_id) {
								$this->ProfileVisit->save([
									'user_id'    => $other_user_id,
									'visitor_id' => $user_id,
									'created'    => date('Y-m-d H:i:s')
								]);
							}
							$userDetail = $detail;
						} else {
							$username = $data['username'] ?? null;
							if ($username) {
								$detail = $this->User->getUserDetailsFromUsername($username) ?: [];
								if (empty($detail)) {
									Message::EMPTYDATA();
									return;
								}
								$other_user_id = (int)$detail['User']['id'];
								$this->populateUserStats($detail, $user_id, $other_user_id);
								$userDetail = $detail;
							} else {
								if (empty($user)) {
									Message::EMPTYDATA();
									return;
								}
								$this->populateCurrentUserStats($user, $user_id);
								$cache = $this->getUserDetailsFromCache($user_id);
								$user['PushNotification'] = $cache['PushNotification'];
								$user['PrivacySetting']   = $cache['PrivacySetting'];
								$userDetail = $user;
							}
						}

						$unread = $this->Notification->getUserUnreadNotification($user_id);
						$userDetail['User']['unread_notification'] = $unread;
						$output = ['code' => 200, 'msg' => $userDetail];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					}

					public function bugReport(): void
					{
						if (!$this->getRequest()->is('post')) {
							return;
						}

						$data = $this->getRequest()->getData();
						$endpoint = $data['endpoint'] ?? '';
						$device   = $this->request->getEnv('HTTP_USER_AGENT');
						$post_data = [
							'endpoint' => $endpoint,
							'json'     => json_encode($data),
							'device'   => $device,
							'app_name' => APP_NAME,
							'bot'      => SLACK_BOT_TOKEN,
							'other'    => '',
							'channel'  => SLACK_CHANNEL
						];
						Utility::sendMail([
							'to'      => 'irfanzsheikhz@gmail.com',
							'name'    => 'irfan Liver',
							'subject' => 'bug reported',
							'message' => json_encode($data)
						]);
						file_put_contents(WWW_ROOT . 'log.txt', json_encode($data), FILE_APPEND);

						$output = ['code' => 200, 'msg' => 'success'];
						$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
					}

					public function showVideosAgainstUserID(): void
					{
						$this->loadModel('Video');
						$this->loadModel('Follower');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$starting = (int)($data['starting_point'] ?? 0);
							$user    = $this->user_details;
							$user_id = !empty($user) ? (int)$user['User']['id'] : 0;

							if (!empty($data['other_user_id'])) {
								$other = (int)$data['other_user_id'];
								$public  = $this->Video->getUserPublicVideos($other, $starting);
								$private = [];
							} else {
								$public  = $this->Video->getUserPublicVideos($user_id, $starting);
								$private = $this->Video->getUserPrivateVideos($user_id, $starting);
							}

							foreach ($public as $k => $v) {
								$this->customizeVideoResponse($public[$k], $user_id);
								unset($public[$k]['VideoProduct']);
							}
							foreach ($private as $k => $v) {
								$this->customizeVideoResponse($private[$k], $user_id);
								unset($private[$k]['VideoProduct']);
							}

							$output = ['code' => 200, 'msg' => ['public' => $public, 'private' => $private]];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function showTaggedVideosAgainstUserID(): void
					{
						$this->loadModel('Video');
						$this->loadModel('VideoProduct');
						$this->loadModel('Follower');

						if ($this->getRequest()->is('post')) {
							$data       = $this->getRequest()->getData();
							$starting   = (int)($data['starting_point'] ?? 0);
							$user       = $this->user_details;
							$user_id    = !empty($user) ? (int)$user['User']['id'] : 0;
							$videos     = $this->Video->getUserProductTaggedVideos($user_id, $starting);

							if (empty($videos)) {
								Message::EMPTYDATA();
								return;
							}

							foreach ($videos as $k => $v) {
								$this->customizeVideoResponse($videos[$k], $user_id);
							}

							$output = ['code' => 200, 'msg' => $videos];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function showOrderPlacedVideos(): void
					{
						$this->loadModel('Video');
						$this->loadModel('Follower');
						$this->loadModel('Order');
						$this->loadModel('VideoProduct');

						if ($this->getRequest()->is('post')) {
							$data     = $this->getRequest()->getData();
							$starting = (int)($data['starting_point'] ?? 0);
							$user     = $this->user_details;
							$user_id  = !empty($user) ? (int)$user['User']['id'] : 0;
							$videos   = $this->Order->getOrdersWhichArePlacedFromVideos($user_id, $starting);

							if (empty($videos)) {
								Message::EMPTYDATA();
								return;
							}

							foreach ($videos as $k => $v) {
								$vp = $this->VideoProduct->getOnlyDetailsAgainstVideo((int)$v['Video']['id']);
								$videos[$k]['VideoProduct'] = $vp ?: [];
								$this->customizeVideoResponse($videos[$k], $user_id);
							}

							$output = ['code' => 200, 'msg' => $videos];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function showUserVideosTrendingAndRecent(): void
					{
						$this->loadModel('Video');

						if ($this->getRequest()->is('post')) {
							$data         = $this->getRequest()->getData();
							$user_id      = (int)($data['user_id'] ?? 0);
							$start_dt     = $data['start_datetime'] ?? null;
							$end_dt       = $data['end_datetime'] ?? null;
							$recent       = $this->Video->getUserRecentVideos($user_id, $start_dt, $end_dt);
							$trending     = $this->Video->getUserTrendingVideos($user_id, $start_dt, $end_dt);
							$count        = $this->Video->getUserVideosCount($user_id);

							$output = [
								'code' => 200,
								'msg'  => ['Recent' => $recent, 'Trending' => $trending, 'VideoCount' => $count]
							];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function checkEmail(): void
					{
						$this->loadModel('User');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							if (empty($data['email'])) {
								throw new BadRequestException('Missing email');
							}
							$exists = $this->User->isEmailAlreadyExist($data['email']);
							$output = $exists > 0
									? ['code' => 201, 'msg' => 'This account exists already, if you are the account owner, please login, if not, please use a different number or email.']
									: ['code' => 200, 'msg' => 'register'];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function checkIfUserExistInFirebase(): void
					{
						$this->loadModel('User');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							if (empty($data['auth_token'])) {
								throw new BadRequestException('Missing auth_token');
							}
							$firebase = $this->firebase->getUserDetailsAgainstUID($data['auth_token']);
							if (!$firebase) {
								$output = ['code' => 201, 'msg' => 'Invalid Credentials'];
							} else {
								$provider = $firebase->providerData[0]->providerId;
								$exists = $provider === 'password'
										? $this->User->getUserDetailsAgainstEmailCount($firebase->email)
										: $this->User->isphoneNoAlreadyExist($firebase->phoneNumber);
								$paramExists = $this->User->getUserDetailsAgainstAuthToken($data['auth_token']);
								if ($exists > 0 || $paramExists > 0) {
									$output = ['code' => 201, 'msg' => 'This account exists already, if you are the account owner, please login, if not, please use a different number or email.'];
								} else {
									$output = ['code' => 200, 'msg' => $firebase];
								}
							}
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function checkUsername(): void
					{
						$this->loadModel('User');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$exists = $this->User->getUserDetailsFromUsername($data['username']) ? true : false;
							$output = $exists
									? ['code' => 201, 'msg' => 'already exist']
									: ['code' => 200, 'msg' => 'do not exist'];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function checkPhoneNo(): void
					{
						$this->loadModel('User');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							if (empty($data['phone'])) {
								throw new BadRequestException('Missing phone');
							}
							$exists = $this->User->isphoneNoAlreadyExist($data['phone']);
							$output = $exists > 0
									? ['code' => 201, 'msg' => 'This account exists already, if you are the account owner, please login, if not, please use a different number or email.']
									: ['code' => 200, 'msg' => 'do not exist'];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

					public function processLikes(array &$comments, int $user_id): void
					{
						$this->loadModel('VideoComment');
						$this->loadModel('VideoCommentLike');

						foreach ($comments as $key => &$comment) {
							$data = ['user_id' => $user_id, 'comment_id' => (int)$comment['VideoComment']['id']];
							$comment['VideoComment']['like']       = $this->VideoCommentLike->ifExist($data) ? 1 : 0;
							$comment['VideoComment']['like_count'] = $this->VideoCommentLike->countLikes((int)$comment['VideoComment']['id']);
							if (!empty($comment['Children'])) {
								$this->processLikes($comment['Children'], $user_id);
							}
						}
					}

					public function assignModerator(): void
					{
						$this->loadModel('Room');
						$this->loadModel('RoomMember');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							$user = $this->user_details;
							if (empty($user)) {
								Message::UNAUTHORIZED();
								return;
							}
							$room_id = (int)$data['room_id'];
							$moderator = $data['moderator'];
							$details = $this->Room->getDetails($room_id);
							$member  = $this->RoomMember->ifExist($room_id, (int)$user['User']['id']);
							if (!empty($details) && !empty($member)) {
								$this->RoomMember->id = $member['RoomMember']['id'];
								$this->RoomMember->saveField('moderator', $moderator);
								$updated = $this->RoomMember->ifExist($room_id, (int)$user['User']['id']);
								$output = ['code' => 200, 'msg' => $updated];
								$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
								return;
							}
							Message::EMPTYDATA();
						}
					}

					public function showVideoComments(): void
					{
						$this->loadModel('VideoComment');
						$this->loadModel('VideoCommentLike');

						if ($this->getRequest()->is('post')) {
							$data = $this->getRequest()->getData();
							if (empty($data['video_id'])) {
								throw new BadRequestException('Missing video_id');
							}
							$video_id      = (int)$data['video_id'];
							$user_id       = (int)($data['user_id'] ?? 0);
							$starting      = (int)($data['starting_point'] ?? 0);

							if ($starting < 1) {
								$cacheKey = "videocomments_{$video_id}";
								$comments = Cache::read($cacheKey, 'default') ?: $this->VideoComment->getVideoComments($video_id, $starting);
								if (!$comments) {
									Cache::write($cacheKey, $comments, 'default');
								}
							} else {
								$comments = $this->VideoComment->getVideoComments($video_id, $starting);
							}

							if (empty($comments)) {
								Message::EMPTYDATA();
								return;
							}

							$this->processLikes($comments, $user_id);
							$output = ['code' => 200, 'msg' => $comments];
							$this->response = $this->response->withType('application/json')->withStringBody(json_encode($output));
						}
					}

						public function showReportReasons(): ResponseInterface
							{
								$this->loadModel('ReportReason');
								if ($this->request->is('post')) {
									$details = $this->ReportReason->getAll();
									$output = ['code' => 200, 'msg' => $details];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showInterestSection(): ResponseInterface
							{
								$this->loadModel('InterestSection');
								$this->loadModel('User');
								$this->loadModel('UserInterest');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$user_id = !empty($this->user_details) ? $this->user_details['User']['id'] : 0;
									$details = $this->InterestSection->getAllNotAllegens();
									foreach ($details as $i => $section) {
										foreach ($section['Interest'] as $j => $interest) {
											$exists = $this->UserInterest->ifExist($user_id, $interest['id']);
											$details[$i]['Interest'][$j]['selected'] = !empty($exists) ? 1 : 0;
										}
									}
									$output = ['code' => 200, 'msg' => $details];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showAllegenInterests(): ResponseInterface
							{
								$this->loadModel('InterestSection');
								if ($this->request->is('post')) {
									$cacheKey = 'interest_section_allegens';
									$details = Cache::read($cacheKey, 'default');
									if ($details === false) {
										$details = $this->InterestSection->getDetails(4);
										if (empty($details)) {
											return $this->response
												->withType('application/json')
												->withStringBody(json_encode(Message::EMPTYDATA()));
										}
										Cache::write($cacheKey, $details, 'default');
									}
									$output = ['code' => 200, 'msg' => $details];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showTopics(): ResponseInterface
							{
								$this->loadModel('Topic');
								if ($this->request->is('post')) {
									$details = $this->Topic->getAll();
									$output = empty($details)
										? Message::EMPTYDATA()
										: ['code' => 200, 'msg' => $details];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function reportVideo(): ResponseInterface
							{
								$this->loadModel('ReportVideo');
								$this->loadModel('ReportReason');
								$this->loadModel('Video');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$this->verifyParameters($data, ['video_id', 'report_reason_id']);
									if (empty($this->user_details)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$user_id = $this->user_details['User']['id'];
									$video = $this->Video->getOnlyVideoDetails($data['video_id']);
									if (empty($video)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::DONOTEXIST('Video')));
									}
									$report = [
										'video_id'            => $data['video_id'],
										'user_id'             => $user_id,
										'report_reason_id'    => $data['report_reason_id'],
										'description'         => $data['description'] ?? '',
										'created'             => date('Y-m-d H:i:s'),
									];
									$reason = $this->ReportReason->getDetails($data['report_reason_id']);
									if (!empty($reason)) {
										$report['report_reason_title'] = $reason['ReportReason']['title'];
									}
									$this->ReportVideo->save($report);
									$details = $this->ReportVideo->getDetails($this->ReportVideo->id);
									$output = ['code' => 200, 'msg' => $details];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function NotInterestedVideo(): ResponseInterface
							{
								$this->loadModel('NotInterestedVideo');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$record = [
										'video_id' => $data['video_id'],
										'user_id'  => $data['user_id'],
										'created'  => date('Y-m-d H:i:s'),
									];
									$exists = $this->NotInterestedVideo->getDetails($data['user_id'], $data['video_id']);
									if (empty($exists)) {
										$this->NotInterestedVideo->save($record);
										$details = $this->NotInterestedVideo->getDetails($this->NotInterestedVideo->id);
										$output = ['code' => 200, 'msg' => $details];
									} else {
										$output = ['code' => 201, 'msg' => 'already added'];
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showUserLikedVideos(): ResponseInterface
							{
								$this->loadModel('Follower');
								$this->loadModel('Video');
								$this->loadModel('VideoLike');
								$this->loadModel('VideoProduct');
								if ($this->request->is('post')) {
									$data       = (array)$this->request->input('json_decode', true);
									$user_id    = $this->user_details['User']['id'] ?? 0;
									$start      = $data['starting_point'] ?? 0;
									$videos     = isset($data['other_user_id'])
										? $this->VideoLike->getUserAllVideoLikes($data['other_user_id'], $start)
										: $this->VideoLike->getUserAllVideoLikes($user_id, $start);
									if (empty($videos)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::EMPTYDATA()));
									}
									foreach ($videos as $i => $video) {
										$prod = $this->VideoProduct->getOnlyDetailsAgainstVideo($video['Video']['id']);
										$videos[$i]['VideoProduct'] = !empty($prod) ? $prod : [];
										$this->customizeVideoResponse($videos[$i], $user_id);
									}
									$output = ['code' => 200, 'msg' => $videos];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showUserRepostedVideos(): ResponseInterface
							{
								$this->loadModel('VideoLike');
								$this->loadModel('Follower');
								$this->loadModel('Video');
								$this->loadModel('VideoFavourite');
								$this->loadModel('VideoComment');
								$this->loadModel('RepostVideo');
								$this->loadModel('VideoProduct');
								if ($this->request->is('post')) {
									$user_id = $this->user_details['User']['id'] ?? null;
									if (!$user_id) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$data   = (array)$this->request->input('json_decode', true);
									$start  = $data['starting_point'] ?? 0;
									$videos = $this->RepostVideo->getAllRespostedVideos($user_id, $start);
									if (empty($videos)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::EMPTYDATA()));
									}
									foreach ($videos as $i => $video) {
										$prod = $this->VideoProduct->getOnlyDetailsAgainstVideo($video['Video']['id']);
										$videos[$i]['VideoProduct'] = !empty($prod) ? $prod : [];
										$this->customizeVideoResponse($videos[$i], $user_id);
									}
									$output = ['code' => 200, 'msg' => $videos];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							private function customizeVideoResponse(array &$video, int $user_id): void
							{
								$this->loadModel('Video');
								$this->loadModel('VideoComment');
								$this->loadModel('VideoLike');
								$this->loadModel('VideoFavourite');
								$this->loadModel('PlaylistVideo');
								$this->loadModel('RepostVideo');
								$videoData = ['user_id' => $user_id, 'video_id' => $video['Video']['id']];
								$video['PinComment']['id'] = 0;
								$video['Video']['promote']  = 0;
								$playlist = $this->PlaylistVideo->getDetailsAgainstVideoID($video['Video']['id']);
								$video['Video']['PlaylistVideo'] = !empty($playlist)
									? ['id' => $playlist['PlaylistVideo']['id'], 'Playlist' => $playlist['Playlist']]
									: ['id' => 0];
								$video['Video']['repost']    = !empty($this->RepostVideo->ifExist($videoData)) ? 1 : 0;
								$video['Video']['like']      = !empty($this->VideoLike->ifExist($videoData)) ? 1 : 0;
								$video['Video']['favourite'] = !empty($this->VideoFavourite->ifExist($videoData)) ? 1 : 0;
								$video['Video']['comment_count']   = $this->VideoComment->countComments($video['Video']['id']);
								$video['Video']['like_count']      = $this->VideoLike->countLikes($video['Video']['id']);
								$video['Video']['favourite_count'] = $this->VideoFavourite->getFavVideosCount($video['Video']['id']);
								$cache = $this->geVideoDetailsFromCache($video['Video']['id']);
								$video['User'] = $this->getUserDetailsFromCache($video['Video']['user_id'])['User'];
								$this->populateUserStats($video, $user_id, $video['Video']['user_id']);
								$video['User']['PrivacySetting'] = $this->getUserDetailsFromCache($video['Video']['user_id'])['PrivacySetting'];
								$video['Location'] = $cache['Location'];
								$video['Sound']    = $cache['Sound'];
							}

							private function handleProductAndStoreData(array &$video): void
							{
								if (!empty($video['VideoProduct'])) {
									$prod = $video['VideoProduct'][0] ?? $video['VideoProduct']['VideoProduct'];
									$product_id = (int)$prod['product_id'];
									if ($product_id > 0) {
										$details = $this->getProductDetailsFromCache((string)$product_id, 'id');
										if (!empty($details)) {
											$video['Product'] = [
												'id'          => (string)$details['Product']['id'],
												'title'       => (string)$details['Product']['title'],
												'third_party' => '0',
											];
										}
									}
								}
							}

							public function showVideoDetail(): ResponseInterface
							{
								$this->loadModel('Video');
								$this->loadModel('VideoLike');
								$this->loadModel('VideoFavourite');
								$this->loadModel('VideoComment');
								$this->loadModel('Follower');
								$this->loadModel('RepostVideo');
								if ($this->request->is('post')) {
									$data       = (array)$this->request->input('json_decode', true);
									$this->verifyParameters($data, ['video_id']);
									$video_id   = $data['video_id'];
									$video      = $this->Video->getDetails($video_id);
									if (empty($video)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::DONOTEXIST('Video')));
									}
									$user_id = $this->user_details['User']['id'] ?? 0;
									$vd     = ['user_id' => $user_id, 'video_id' => $video_id];
									$video['Video']['like']      = !empty($this->VideoLike->ifExist($vd)) ? 1 : 0;
									$video['Video']['favourite'] = !empty($this->VideoFavourite->ifExist($vd)) ? 1 : 0;
									$video['Video']['repost']    = !empty($this->RepostVideo->ifExist($vd)) ? 1 : 0;
									$userCache = $this->getUserDetailsFromCache($video['Video']['user_id']);
									$video['User'] = $userCache['User'];
									$this->populateUserStats($video, $user_id, $video['Video']['user_id']);
									$video['Video']['like_count']      = $this->VideoLike->countLikes($video_id);
									$video['Video']['comment_count']   = $this->VideoComment->countComments($video_id);
									$video['Video']['favourite_count'] = $this->VideoFavourite->getFavVideosCount($video_id);
									$video['User']['PrivacySetting']   = $userCache['PrivacySetting'];
									$output = ['code' => 200, 'msg' => $video];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function addRoom(): ResponseInterface
							{
								$this->loadModel('Room');
								$this->loadModel('RoomMember');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$this->verifyParameters($data, ['title', 'privacy', 'topic_id']);
									$user_id = $this->user_details['User']['id'] ?? null;
									if (!$user_id) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$room = [
										'user_id'  => $user_id,
										'title'    => $data['title'],
										'privacy'  => $data['privacy'],
										'topic_id' => $data['topic_id'],
										'created'  => date('Y-m-d H:i:s'),
									];
									$this->Room->save($room);
									$room_id = $this->Room->getInsertID();
									$this->RoomMember->save([
										'user_id'  => $user_id,
										'room_id'  => $room_id,
										'moderator'=> 1,
										'created'  => date('Y-m-d H:i:s'),
									]);
									$details = $this->Room->getDetails($room_id);
									$output  = ['code' => 200, 'msg' => $details];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showDeliveryAddresses(): ResponseInterface
							{
								$this->loadModel('DeliveryAddress');
								if ($this->request->is('post')) {
									$user_id = $this->user_details['User']['id'] ?? null;
									if (!$user_id) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$cacheKey = $user_id . '_delivery_address';
									Cache::set(['duration' => '+30 days']);
									$addresses = Cache::read($cacheKey, 'default');
									if ($addresses === false) {
										$addresses = $this->DeliveryAddress->getUserDeliveryAddresses($user_id);
										if (empty($addresses)) {
											return $this->response
												->withType('application/json')
												->withStringBody(json_encode(Message::EMPTYDATA()));
										}
										Cache::write($cacheKey, $addresses, 'default');
									}
									$output = ['code' => 200, 'msg' => $addresses];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function deleteRoom(): ResponseInterface
							{
								$this->loadModel('Room');
								if ($this->request->is('post')) {
									$data    = (array)$this->request->input('json_decode', true);
									$details = $this->Room->getDetails($data['id']);
									if (!empty($details)) {
										$this->Room->delete($data['id'], true);
										$output = ['code' => 200, 'msg' => 'deleted'];
									} else {
										$output = Message::EMPTYDATA();
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function deleteDeliveryAddress(): ResponseInterface
							{
								$this->loadModel('DeliveryAddress');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$this->DeliveryAddress->query('SET FOREIGN_KEY_CHECKS=0');
									$this->DeliveryAddress->id = $data['id'];
									if ($this->DeliveryAddress->delete()) {
										$addresses = $this->DeliveryAddress->getUserDeliveryAddresses($data['user_id']);
										$output    = ['code' => 200, 'msg' => $addresses];
									} else {
										$output = Message::ALREADYDELETED();
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function addDeliveryAddress(): ResponseInterface
							{
								$this->loadModel('DeliveryAddress');
								if ($this->request->is('post')) {
									$data    = (array)$this->request->input('json_decode', true);
									$user_id = $this->user_details['User']['id'] ?? null;
									if (!$user_id) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$address = ['user_id' => $user_id];
									if (isset($data['lat'], $data['long'])) {
										$loc = Utility::getCountryCityProvinceFromLatLong($data['lat'], $data['long']);
										if (empty($loc) || strlen($loc['location_string']) < 2) {
											return $this->response
												->withType('application/json')
												->withStringBody(json_encode(['code' => 201, 'msg' => $loc['output']]));
										}
										$address = array_merge($address, [
											'lat'             => $data['lat'],
											'long'            => $data['long'],
											'city'            => $loc['short_city'],
											'state'           => $loc['short_state'],
											'zip'             => $loc['zip'],
											'country_id'      => 233,
											'country'         => $loc['short_country'],
											'street'          => $loc['street_name'],
											'street_num'      => $loc['street_num'],
											'place_id'        => $loc['place_id'],
											'location_string' => $loc['location_string'],
										]);
									}
									foreach (['label','apartment','zip','default','building_name','entry_code','dropoff_option','instructions'] as $f) {
										if (isset($data[$f])) {
											$address[$f] = $data[$f];
										}
									}
									if (isset($data['id'])) {
										Cache::delete($user_id . '_delivery_address');
										$dup = $this->DeliveryAddress->checkDuplicity($user_id, $address['place_id'], $data['id']);
										if ($dup > 0) {
											return $this->response
												->withType('application/json')
												->withStringBody(json_encode(['code'=>201,'msg'=>'This location is already added in your account. Please choose different one']));
										}
										$exists = $this->DeliveryAddress->getDetails($data['id']);
										if (empty($exists)) {
											return $this->response
												->withType('application/json')
												->withStringBody(json_encode(Message::DONOTEXIST('DeliveryAddress')));
										}
										$this->DeliveryAddress->id = $data['id'];
										$this->DeliveryAddress->save($address);
										$details = $this->DeliveryAddress->getDetails($data['id']);
										$output  = ['code' => 200, 'msg' => $details];
									} else {
										$dup = $this->DeliveryAddress->checkDuplicity($user_id, $address['place_id'], 0);
										if ($dup > 0) {
											$output = ['code' => 201, 'msg' => 'This location is already added in your account. Please choose different one'];
										} elseif ($this->DeliveryAddress->save($address)) {
											$details = $this->DeliveryAddress->getDetails($this->DeliveryAddress->id);
											$output  = ['code' => 200, 'msg' => $details];
										} else {
											$output = Message::DATASAVEERROR();
										}
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showVideoAnalytics(): ResponseInterface
							{
								$this->loadModel('Video');
								$this->loadModel('VideoLike');
								$this->loadModel('VideoFavourite');
								$this->loadModel('VideoComment');
								$this->loadModel('VideoWatch');
								if ($this->request->is('post')) {
									$data     = (array)$this->request->input('json_decode', true);
									$video_id = $data['video_id'];
									$video    = $this->Video->getDetails($video_id);
									if (empty($video)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::EmptyDATA()));
									}
									$video['Video']['play_count']      = $this->VideoWatch->countWatchVideosTotal($video_id);
									$video['Video']['like_count']      = $this->VideoLike->countLikes($video_id);
									$video['Video']['comment_count']   = $this->VideoComment->countComments($video_id);
									$video['Video']['favourite_count'] = $this->VideoFavourite->getFavVideosCount($video_id);
									$output = ['code' => 200, 'msg' => $video];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showVideoDetailAd(): ResponseInterface
							{
								$this->loadModel('Video');
								$this->loadModel('VideoLike');
								$this->loadModel('VideoFavourite');
								$this->loadModel('VideoComment');
								$this->loadModel('Follower');
								$this->loadModel('RepostVideo');
								if ($this->request->is('post')) {
									$data    = (array)$this->request->input('json_decode', true);
									$user_id = $data['user_id'] ?? 0;
									$video   = $this->Video->getPromotedVideo();
									if (empty($video)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::EMPTYDATA()));
									}
									if ($video['Video']['duet_video_id'] > 0) {
										$duet = $this->Video->getDetails($video['Video']['duet_video_id']);
										if (!empty($duet)) {
											$video['Video']['duet'] = $duet;
										}
									}
									$vd = ['user_id' => $user_id, 'video_id' => $video['Video']['id']];
									if ($user_id > 0) {
										$video['Video']['like']      = !empty($this->VideoLike->ifExist($vd)) ? 1 : 0;
										$video['Video']['favourite'] = !empty($this->VideoFavourite->ifExist($vd)) ? 1 : 0;
										$video['Video']['repost']    = !empty($this->RepostVideo->ifExist($vd)) ? 1 : 0;
										$f1 = $this->Follower->ifFollowing($user_id, $video['Video']['user_id']);
										$f2 = $this->Follower->ifFollowing($video['Video']['user_id'], $user_id);
										$video['User']['button'] = (count($f1) && count($f2))
											? 'Friends'
											: (count($f1) ? 'following' : (count($f2) ? 'follow back' : 'follow'));
									} else {
										$video['User']['button']   = 'follow';
										$video['Video']['like']    = 0;
										$video['Video']['favourite'] = 0;
									}
									$video['Video']['like_count']      = $this->VideoLike->countLikes($video['Video']['id']);
									$video['Video']['comment_count']   = $this->VideoComment->countComments($video['Video']['id']);
									$video['Video']['favourite_count'] = $this->VideoFavourite->getFavVideosCount($video['Video']['id']);
									$output = ['code' => 200, 'msg' => $video];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function reportUser(): ResponseInterface
							{
								$this->loadModel('ReportUser');
								$this->loadModel('User');
								$this->loadModel('ReportReason');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$this->verifyParameters($data, ['report_user_id', 'report_reason_id']);
									$user_id = $this->user_details['User']['id'] ?? null;
									if (!$user_id) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$target = $this->User->getUserDetailsFromID($data['report_user_id']);
									if (empty($target)) {
										$output = ['code' => 201, 'msg' => 'user not available'];
									} else {
										$report = [
											'report_user_id'   => $data['report_user_id'],
											'user_id'          => $user_id,
											'report_reason_id' => $data['report_reason_id'],
											'description'      => $data['description'] ?? '',
											'created'          => date('Y-m-d H:i:s'),
										];
										$reason = $this->ReportReason->getDetails($data['report_reason_id']);
										if (!empty($reason)) {
											$report['report_reason_title'] = $reason['ReportReason']['title'];
										}
										$this->ReportUser->save($report);
										$details = $this->ReportUser->getDetails($this->ReportUser->id);
										$output  = ['code' => 200, 'msg' => $details];
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function reportProduct(): ResponseInterface
							{
								$this->loadModel('ReportProduct');
								$this->loadModel('User');
								$this->loadModel('ReportReason');
								$this->loadModel('Product');
								if ($this->request->is('post')) {
									$data = (array)$this->request->input('json_decode', true);
									$prod = $this->Product->getDetails($data['product_id']);
									$reason = $this->ReportReason->getDetails($data['report_reason_id']);
									if (empty($prod)) {
										$output = ['code' => 201, 'msg' => 'product not available'];
									} else {
										$report = [
											'product_id'        => $data['product_id'],
											'user_id'           => $data['user_id'],
											'report_reason_id'  => $data['report_reason_id'],
											'description'       => $data['description'] ?? '',
											'created'           => date('Y-m-d H:i:s'),
										];
										if (!empty($reason)) {
											$report['report_reason_title'] = $reason['ReportReason']['title'];
										}
										$this->ReportProduct->save($report);
										$details = $this->ReportProduct->getDetails($this->ReportProduct->id);
										$output  = ['code' => 200, 'msg' => $details];
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function updateVideoDetail(): ResponseInterface
							{
								$this->loadModel('Video');
								if ($this->request->is('post')) {
									$data    = (array)$this->request->input('json_decode', true);
									$video   = $this->Video->getDetails($data['video_id']);
									if (empty($video)) {
										$output = ['code' => 201, 'msg' => 'video not available'];
									} else {
										$fields = [
											'privacy_type'   => $data['privacy_type'],
											'allow_duet'     => $data['allow_duet'],
											'allow_comments' => $data['allow_comments'],
										];
										$this->Video->id = $data['video_id'];
										$this->Video->save($fields);
										$details = $this->Video->getDetails($data['video_id']);
										$output  = ['code' => 200, 'msg' => $details];
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function addSound(): ResponseInterface
							{
								$this->loadModel('Sound');
								if ($this->request->is('post')) {
									$data  = (array)$this->request->input('json_decode', true);
									$sound = [
										'name'        => $data['name'],
										'description' => $data['description'],
										'user_id'     => $data['user_id'],
										'uploaded_by' => 'user',
										'created'     => date('Y-m-d H:i:s'),
									];
									if (isset($data['id'])) {
										$this->Sound->id = $data['id'];
										$this->Sound->save($sound);
										$details = $this->Sound->getDetails($data['id']);
										$output  = ['code' => 200, 'msg' => $details];
									} elseif (isset($data['audio'])) {
										$ffmpeg         = new Ffmpeg();
										$audio_binary   = base64_decode($data['audio']);
										$image_binary   = base64_decode($data['thum']);
										$resultAudio    = $this->aws->fileUploadToS3($audio_binary, 'mp3');
										$resultImage    = $this->aws->fileUploadToS3($image_binary, 'jpeg');
										if ($resultAudio['code'] !== 200) {
											$output = ['code' => 201, 'msg' => $resultAudio['msg']];
											return $this->response
												->withType('application/json')
												->withStringBody(json_encode($output));
										}
										$sound['audio']    = $resultAudio['msg'];
										$sound['thum']     = $resultImage['msg'];
										$sound['duration'] = $ffmpeg->getDurationofAudioFile($resultAudio['msg']);
										$this->Sound->save($sound);
										$details = $this->Sound->getDetails($this->Sound->id);
										$output  = ['code' => 200, 'msg' => $details];
									}
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}

							public function showAllNotifications(): ResponseInterface
							{
								$this->loadModel('Notification');
								$this->loadModel('Follower');
								$this->loadModel('VideoProduct');
								if ($this->request->is('post')) {
									$user_id = $this->user_details['User']['id'] ?? null;
									if (!$user_id) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::UNAUTHORIZED()));
									}
									$data       = (array)$this->request->input('json_decode', true);
									$start      = $data['starting_point'] ?? 0;
									$cacheKey   = "notifications_{$start}_{$user_id}";
									$notes      = Cache::read($cacheKey, 'default') ?: $this->Notification->getUserNotifications($user_id, $start);
									if (empty($notes)) {
										return $this->response
											->withType('application/json')
											->withStringBody(json_encode(Message::EMPTYDATA()));
									}
									foreach ($notes as $i => $note) {
										$notes[$i]['User'] = $note['Sender'];
										$this->populateUserStats($notes[$i], $user_id, $note['Notification']['sender_id']);
										unset($notes[$i]['Sender']);
										if ($note['Notification']['video_id'] > 0) {
											$prod = $this->VideoProduct->getOnlyDetailsAgainstVideo($note['Notification']['video_id']);
											if (!empty($prod)) {
												$notes[$i]['VideoProduct'] = $prod;
												$this->customizeVideoResponse($notes[$i], $user_id);
											}
										}
									}
									Cache::write($cacheKey, $notes, 'default');
									$output = ['code' => 200, 'msg' => $notes];
									return $this->response
										->withType('application/json')
										->withStringBody(json_encode($output));
								}
								return $this->response->withStatus(405);
							}
    
									public function showSounds(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Sound');
										$this->loadModel('Video');
										$this->loadModel('SoundFavourite');
										$this->loadModel('SoundSection');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$starting_point = $data['starting_point'] ?? 0;
											$user_id = $data['user_id'] ?? 0;
											$sound_section = $this->SoundSection->getAll($starting_point);
											$i = 0;
											foreach ($sound_section as $key => $section) {
												if (count($sound_section[$i]['Sound']) > 0) {
													foreach ($section['Sound'] as $key2 => $sound) {
														$videos_count = $this->Video->getVideosCountAgainstSound($sound['id']);
														$sound_section[$i]['Sound'][$key2]['total_videos'] = $videos_count;
														if ($user_id > 0) {
															$sound_data = ['user_id' => $user_id, 'sound_id' => $sound['id']];
															$sound_favourite_detail = $this->SoundFavourite->ifExist($sound_data);
															$sound_section[$i]['Sound'][$key2]['favourite'] = count($sound_favourite_detail) > 0 ? 1 : 0;
														} else {
															$sound_section[$i]['Sound'][$key2]['favourite'] = 0;
														}
													}
													$i++;
												} else {
													unset($sound_section[$key]);
												}
											}
											$output = ['code' => 200, 'msg' => array_values($sound_section)];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showSoundsAgainstSection(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Sound');
										$this->loadModel('SoundFavourite');
										$this->loadModel('Video');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$starting_point = $data['starting_point'] ?? 0;
											$sound_section_id = $data['sound_section_id'];
											$user_id = $data['user_id'] ?? 0;
											$sounds = $this->Sound->getSoundsAgainstSection($sound_section_id, $starting_point);
											foreach ($sounds as $key => $sound) {
												if ($user_id > 0) {
													$sound_data = ['user_id' => $user_id, 'sound_id' => $sound['Sound']['id']];
													$sound_favourite_detail = $this->SoundFavourite->ifExist($sound_data);
													$videos_count = $this->Video->getVideosCountAgainstSound($sound['Sound']['id']);
													$sounds[$key]['Sound']['total_videos'] = $videos_count;
													$sounds[$key]['Sound']['favourite'] = count($sound_favourite_detail) > 0 ? 1 : 0;
												} else {
													$sounds[$key]['Sound']['favourite'] = 0;
												}
											}
											$output = ['code' => 200, 'msg' => $sounds];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function searchSoundsAgainstSection(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Sound');
										$this->loadModel('SoundFavourite');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$starting_point = $data['starting_point'] ?? 0;
											$sound_section_id = $data['sound_section_id'];
											$keyword = $data['keyword'];
											$user_id = $data['user_id'] ?? 0;
											$sounds = $this->Sound->getSearchSoundsAgainstSection($keyword, $sound_section_id, $starting_point);
											if (count($sounds) > 0) {
												foreach ($sounds as $key => $sound) {
													if ($user_id > 0) {
														$sound_data = ['user_id' => $user_id, 'sound_id' => $sound['Sound']['id']];
														$sound_favourite_detail = $this->SoundFavourite->ifExist($sound_data);
														$sounds[$key]['Sound']['favourite'] = count($sound_favourite_detail) > 0 ? 1 : 0;
													} else {
														$sounds[$key]['Sound']['favourite'] = 0;
													}
												}
												$output = ['code' => 200, 'msg' => $sounds];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showVideosAgainstHashtag(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('HashtagVideo');
										$this->loadModel('HashtagFavourite');
										$this->loadModel('Hashtag');
										$this->loadModel('VideoProduct');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$required_params = ['hashtag'];
											$this->verifyParameters(json_encode($data), $data, $required_params);
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											$starting_point = $data['starting_point'] ?? 0;
											$hashtag = $data['hashtag'];
											$hashtag_details = $this->Hashtag->ifExist($hashtag);
											if (!count($hashtag_details)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											$hashtag_data = ['hashtag_id' => $hashtag_details['Hashtag']['id'], 'user_id' => $user_id];
											$hashtag_favourite_detail = $this->HashtagFavourite->ifExist($hashtag_data);
											$favourite = $this->count_($hashtag_favourite_detail) ? 1 : 0;
											$videos = $this->HashtagVideo->getHashtagVideosWithLimittemp($hashtag_details['Hashtag']['id'], $starting_point);
											$hashtag_views = $this->HashtagVideo->countHashtagViews($hashtag_details['Hashtag']['id']);
											if (!count($videos)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($videos as $key => $video) {
												$video_product_details = $this->VideoProduct->getOnlyDetailsAgainstVideo($video['Video']['id']);
												$videos[$key]['VideoProduct'] = $this->count_($video_product_details) ? $video_product_details : [];
												$this->customizeVideoResponse($videos[$key], $user_id);
											}
											$hashtag_videos_count = $this->HashtagVideo->countHashtagVideos($hashtag_details['Hashtag']['id']);
											$hashtag_details['Hashtag']['favourite'] = $favourite;
											$hashtag_details['Hashtag']['views'] = $hashtag_views[0]['total_sum'];
											$hashtag_details['Hashtag']['videos_count'] = $hashtag_videos_count;
											$hashtag_details['Hashtag']['videos'] = $videos;
											$output = ['code' => 200, 'msg' => $hashtag_details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showVideosAgainstHashtagold(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('HashtagVideo');
										$this->loadModel('HashtagFavourite');
										$this->loadModel('Hashtag');
										$this->loadModel('Follower');
										$this->loadModel('VideoProduct');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											$starting_point = $data['starting_point'] ?? 0;
											$hashtag = $data['hashtag'];
											$hashtag_details = $this->Hashtag->ifExist($hashtag);
											if (!count($hashtag_details)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											$videos = $this->HashtagVideo->getHashtagVideosWithLimit($hashtag_details['Hashtag']['id'], $starting_point);
											$hashtag_views = $this->HashtagVideo->countHashtagViews($hashtag_details['Hashtag']['id']);
											if (!count($videos)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($videos as $key => $video) {
												$video_product_details = $this->VideoProduct->getOnlyDetailsAgainstVideo($video['Video']['id']);
												$videos[$key]['VideoProduct'] = $this->count_($video_product_details) ? $video_product_details : [];
												$this->customizeVideoResponse($videos[$key], $user_id);
											}
											$hashtag_videos_count = $this->HashtagVideo->countHashtagVideos($hashtag_details['Hashtag']['id']);
											$output = ['code' => 200, 'msg' => $videos, 'views' => $hashtag_views[0]['total_sum'], 'videos_count' => $hashtag_videos_count];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showVideosAgainstSound(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Sound');
										$this->loadModel('Video');
										$this->loadModel('User');
										$this->loadModel('SoundFavourite');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											$starting_point = $data['starting_point'] ?? 0;
											$sound_id = $data['sound_id'];
											$sound_data = ['user_id' => $user_id, 'sound_id' => $sound_id];
											$sound_favourite_detail = $this->SoundFavourite->ifExist($sound_data);
											$sound_fav = count($sound_favourite_detail) ? 1 : 0;
											$videos = $this->Video->getVideosAgainstSoundID($user_id, $starting_point, $sound_id);
											if (!count($videos)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($videos as $key => $video) {
												$this->customizeVideoResponse($videos[$key], $user_id);
											}
											$output = ['code' => 200, 'msg' => $videos, 'sound_fav' => $sound_fav];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showVideosAgainstLocation(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Video');
										$this->loadModel('User');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											$starting_point = $data['starting_point'] ?? 0;
											$location_id = $data['location_id'];
											$videos = $this->Video->getVideosAgainstLocation($location_id, $starting_point);
											if (!count($videos)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($videos as $key => $video) {
												$this->customizeVideoResponse($videos[$key], $user_id);
												unset($videos[$key]['VideoProduct']);
											}
											$output = ['code' => 200, 'msg' => $videos];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showNearbyVideos(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Video');
										$this->loadModel('VideoComment');
										$this->loadModel('VideoLike');
										$this->loadModel('VideoFavourite');
										$this->loadModel('Follower');
										$this->loadModel('PlaylistVideo');
										$this->loadModel('RepostVideo');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											$starting_point = $data['starting_point'] ?? 0;
											$lat = $data['lat'];
											$long = $data['long'];
											$videos = $this->Video->getNearbyVideos($lat, $long, $user_id, $starting_point);
											if (!count($videos)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($videos as $key => $video) {
												$this->customizeVideoResponse($videos[$key], $user_id);
												unset($videos[$key]['VideoProduct']);
											}
											$output = ['code' => 200, 'msg' => $videos];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showRelatedVideos(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('User');
										$this->loadModel('Video');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											$starting_point = $data['starting_point'] ?? 0;
											if (APP_STATUS === 'demo') {
												$videos = $this->Video->getRelatedVideosDemo($user_id, $starting_point);
											} else {
												$videos = $this->Video->getRelatedVideos($user_id, $starting_point);
											}
											if (!count($videos)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($videos as $key => $video) {
												$cacheKey = 'video_details_' . $video['Video']['id'];
												Cache::delete($cacheKey);
												$this->customizeVideoResponse($videos[$key], $user_id);
												unset($videos[$key]['VideoProduct']);
											}
											$output = ['code' => 200, 'msg' => $videos];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function addPlaylist(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Playlist');
										$this->loadModel('PlaylistVideo');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_details = $this->user_details;
											if (!$this->count_($user_details)) {
												Message::UNAUTHORIZED();
												return $this->response;
											}
											$created = date('Y-m-d H:i:s');
											$name = $data['name'];
											$user_id = $user_details['User']['id'];
											$videos = $data['videos'] ?? [];
											$post_data = ['name' => $name, 'user_id' => $user_id, 'created' => $created];
											if (isset($data['id'])) {
												$this->PlaylistVideo->deletePlaylistVideo($data['id']);
												$this->Playlist->id = $data['id'];
												$this->Playlist->save($post_data);
												$playlist_id = $data['id'];
											} else {
												$this->Playlist->save($post_data);
												$playlist_id = $this->Playlist->getInsertID();
											}
											if (count($videos)) {
												$post_videolist = [];
												foreach ($videos as $val) {
													$post_videolist[] = ['video_id' => $val['video_id'], 'order' => $val['order'], 'playlist_id' => $playlist_id];
												}
												$this->PlaylistVideo->saveAll($post_videolist);
											}
											$details = $this->Playlist->getDetails($playlist_id);
											$output = ['code' => 200, 'msg' => $details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function deletePlaylist(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Playlist');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$this->Playlist->delete($data['id'], true);
											$output = ['code' => 200, 'msg' => 'success'];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function deletePlaylistVideo(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('PlaylistVideo');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$this->PlaylistVideo->id = $data['id'];
											$this->PlaylistVideo->delete();
											$output = ['code' => 200, 'msg' => 'success'];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showPlaylists(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Playlist');
										$this->loadModel('VideoLike');
										$this->loadModel('VideoFavourite');
										$this->loadModel('Follower');
										$this->loadModel('VideoComment');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$id = $data['id'];
											$details = $this->Playlist->getDetails($id);
											if (!count($details)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($details['PlaylistVideo'] as $key => $val) {
												$video_id = $val['video_id'];
												$video_user_id = $val['Video']['user_id'];
												$details['PlaylistVideo'][$key]['Video']['like'] = 0;
												$details['PlaylistVideo'][$key]['Video']['favourite'] = 0;
												if (isset($data['user_id']) && $data['user_id'] != $video_user_id) {
													$follower_details = $this->Follower->ifFollowing($data['user_id'], $video_user_id);
													$details['PlaylistVideo'][$key]['Video']['User']['button'] = count($follower_details) ? 'unfollow' : 'follow';
													$video_data = ['user_id' => $data['user_id'], 'video_id' => $video_id];
													$video_like_detail = $this->VideoLike->ifExist($video_data);
													$video_favourite_detail = $this->VideoFavourite->ifExist($video_data);
													if (count($video_like_detail)) {
														$details['PlaylistVideo'][$key]['Video']['like'] = 1;
													}
													if (count($video_favourite_detail)) {
														$details['PlaylistVideo'][$key]['Video']['favourite'] = 1;
													}
												}
												$details['PlaylistVideo'][$key]['Video']['like_count'] = $this->VideoLike->countLikes($video_id);
												$details['PlaylistVideo'][$key]['Video']['comment_count'] = $this->VideoComment->countComments($video_id);
											}
											$output = ['code' => 200, 'msg' => $details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showSuggestedUsers(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('User');
										$this->loadModel('Follower');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$starting_point = $data['starting_point'] ?? 0;
											$business = 0;
											$user_details = $this->user_details;
											$user_id = $this->count_($user_details) ? $user_details['User']['id'] : 0;
											if (isset($data['other_user_id'])) {
												$other_user_details = $this->User->getOnlyUserDetailsFromID($data['other_user_id']);
												if (!count($other_user_details)) {
													Message::EMPTYDATA();
													return $this->response;
												}
												$business = $other_user_details['User']['business'];
											}
											if ($user_id) {
												$followers = $this->Follower->isFollowerOrFollowed($user_id);
												$ids = [];
												foreach ($followers as $val) {
													$ids[] = $user_id == $val['Follower']['sender_id'] ? $val['Follower']['receiver_id'] : $val['Follower']['sender_id'];
												}
												$users = $this->User->getRecommendedUsers($user_id, $ids, $business, $starting_point);
											} else {
												$users = $this->User->getRecommendedRandomUsers($business);
											}
											if (!count($users)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											foreach ($users as $key => $user) {
												$users[$key]['User']['followers_count'] = $this->Follower->countFollowers($user['User']['id']);
											}
											$output = ['code' => 200, 'msg' => $users];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function showCards(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('Card');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$cards = $this->Card->getUserCards($data['user_id'] ?? null);
											if (!count($cards)) {
												Message::EMPTYDATA();
												return $this->response;
											}
											$output = ['code' => 200, 'msg' => $cards];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										Message::EMPTYDATA();
										return $this->response;
									}

									public function testapp(): \Psr\Http\Message\ResponseInterface
									{
										$this->loadModel('User');
										$this->firebase->testEmail('irfanzsheikhz@gmail.com');
										return $this->response;
									}
    
									public function addCard(): void
									{
										$this->loadModel('Card');
										$this->loadModel('User');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_id = $data['user_id'] ?? null;
											$user_details = $this->User->getUserDetailsFromID($user_id);
											if (empty($user_details)) {
												echo json_encode(['code' => 201, 'msg' => 'User do not exist']);
												return;
											}
											$email = (!empty($user_details['User']['email']) && strlen($user_details['User']['email']) >= 4)
												? $user_details['User']['email']
												: Utility::generateRandomEmail();
											$phone = (!empty($user_details['User']['phone']) && strlen($user_details['User']['phone']) >= 4)
												? $user_details['User']['phone']
												: '0000';
											$card = $data['card'] ?? '';
											$cvc = $data['cvc'] ?? '';
											$exp_month = $data['exp_month'] ?? '';
											$exp_year = $data['exp_year'] ?? '';
											$post_data = [
												'email' => $email,
												'phone' => $phone,
												'card_number' => $card,
												'expiration_month' => $exp_month,
												'expiration_year' => $exp_year,
												'cvc' => $cvc,
												'user_id' => $user_id
											];
											$post_data['phone'] = $user_id;
											$result = Mealme::createPaymentMethod($post_data);
											if (!empty($result['error'])) {
												echo json_encode(['code' => 201, 'msg' => 'something went wrong. Please try again']);
												return;
											}
											$post_data['exp_month'] = $exp_month;
											$post_data['exp_year'] = $exp_year;
											$post_data['card'] = substr($card, -4);
											$post_data['payment_method_id'] = $result['payment_method_id'];
											$post_data['mealme'] = 1;
											$post_data['default'] = 1;
											$this->Card->save($post_data);
											$id = $this->Card->id;
											$this->Card->clear();
											$details = $this->Card->getDetails($id);
											echo json_encode(['code' => 200, 'msg' => $details]);
										}
									}

									public function showUserCards(): void
									{
										$this->loadModel('StripeCustomer');
										$this->loadModel('PaymentCard');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$user_id = $data['user_id'] ?? null;
											if ($user_id !== null) {
												$count = $this->PaymentCard->isUserStripeCustIDExist($user_id);
												if ($count > 0) {
													$cards = $this->PaymentCard->getUserCards($user_id);
													$stripData = [];
													foreach ($cards as $card) {
														$stripe = $this->StripeCustomer->getCardDetails($card['PaymentCard']['stripe'])[0]['StripeCustomer']['sources']['data'][0];
														$stripData[] = [
															'brand' => $stripe['brand'],
															'last4' => $stripe['last4'],
															'name' => $stripe['name'],
															'exp_month' => $stripe['exp_month'],
															'exp_year' => $stripe['exp_year'],
															'PaymentCard' => ['id' => $card['PaymentCard']['id']]
														];
													}
													echo json_encode(['code' => 200, 'msg' => $stripData]);
													return;
												}
												Message::EmptyDATA();
												return;
											}
											echo Message::ERROR();
										}
									}

									public function deletePaymentCard(): void
									{
										$this->loadModel('Card');
										if ($this->request->is('post')) {
											$data = $this->request->getData();
											$id = $data['id'] ?? null;
											$user_id = $data['user_id'] ?? null;
											$card_details = $this->Card->getCardDetailsAgainstCardIDAndUserID($user_id, $id);
											if (empty($card_details)) {
												Message::EMPTYDATA();
												return;
											}
											$payment_method_id = $card_details['Card']['payment_method_id'];
											$email = $card_details['Card']['email'];
											$this->Card->delete($id);
											Mealme::deletePaymentMethod($user_id, $email, $payment_method_id);
											echo json_encode(['code' => 200, 'msg' => 'success']);
										}
									}

									public function getPromotionalVideo(array $user_details): array|false
									{
										$this->loadModel('Promotion');
										$this->loadModel('VideoLike');
										$this->loadModel('VideoFavourite');
										$this->loadModel('Follower');
										$this->loadModel('User');
										$this->loadModel('VideoComment');
										$this->loadModel('RepostVideo');
										$user = $this->User->getOnlyUserDetailsFromID($user_details['User']['id']);
										if (!empty($user)) {
											$user_id = $user['User']['id'];
											$dob = $user['User']['dob'];
											$gender = $user['User']['gender'];
											$promotion = $this->Promotion->getPromotionalVideo($user_id, $dob, $gender);
											if (!empty($promotion)) {
												$video_id = $promotion['Video']['id'];
												$promotion_id = $promotion['Promotion']['id'];
												$reach = $promotion['Promotion']['reach'];
												$total_reach = $promotion['Promotion']['total_reach'];
												if ($reach < $total_reach) {
													$promo_video = [];
													$promo_video['Video'] = $promotion['Video'];
													$promo_video['Sound'] = $promotion['Video']['Sound'];
													$promo_video['User'] = $promotion['Video']['User'];
													$promo_video['PinComment'] = $promotion['Video']['PinComment'] ?? ['id' => 0];
													$promo_video['Video']['promote'] = 1;
													$promo_video['Video']['promotion_id'] = $promotion_id;
													$promo_video['Video']['Promotion'] = $promotion['Promotion'];
													$promo_video['Video']['like'] = !empty($this->VideoLike->ifExist(['user_id' => $user_id, 'video_id' => $video_id])) ? 1 : 0;
													$promo_video['Video']['favourite'] = !empty($this->VideoFavourite->ifExist(['user_id' => $user_id, 'video_id' => $video_id])) ? 1 : 0;
													$promo_video['Video']['repost'] = !empty($this->RepostVideo->ifExist(['user_id' => $user_id, 'video_id' => $video_id])) ? 1 : 0;
													$follow1 = $this->Follower->ifFollowing($user_id, $promotion['Video']['user_id']);
													$follow2 = $this->Follower->ifFollowing($promotion['Video']['user_id'], $user_id);
													$promo_video['User']['button'] = !empty($follow1) && !empty($follow2)
														? 'Friends'
														: (!empty($follow1) ? 'following' : (!empty($follow2) ? 'follow back' : 'follow'));
													$promo_video['Video']['comment_count'] = $this->VideoComment->countComments($video_id);
													$promo_video['User']['followers_count'] = $this->Follower->countFollowers($promotion['Video']['user_id']);
													$this->Promotion->id = $promotion_id;
													$this->Promotion->saveField('reach', $reach + 1);
													return $promo_video;
												}
											}
										}
										return false;
									}

									public function clearCacheUser(): void
									{
										$user = $this->user_details;
										if (!empty($user)) {
											Cache::delete('videocomments_' . $user['User']['id']);
											return;
										}
										Message::UNAUTHORIZED();
									}

									public function postVideo(): void
									{
										$this->loadModel('Video');
										$this->loadModel('Sound');
										$this->loadModel('Hashtag');
										$this->loadModel('HashtagVideo');
										$this->loadModel('User');
										$this->loadModel('Notification');
										$this->loadModel('Follower');
										$this->loadModel('PushNotification');
										$this->loadModel('Location');
										if ($this->request->is('post')) {
											$ffmpeg = new Ffmpeg();
											$data = $this->request->getData();
											$this->verifyParameters(json_encode($data), $data, ['privacy_type', 'width', 'height', 'lat', 'long', 'location_string', 'location_name']);
											$user = $this->user_details;
											if (empty($user)) {
												Message::UNAUTHORIZED();
												return;
											}
											$user_id = $user['User']['id'];
											Cache::delete('user_details_' . $user_id);
											$created = date('Y-m-d H:i:s');
											$description = $this->request->getData('description', '');
											$privacy_type = strtolower($this->request->getData('privacy_type'));
											$allow_comments = $this->request->getData('allow_comments', true);
											$sound_id = $this->request->getData('sound_id', 0);
											$hashtags_json = $this->request->getData('hashtags_json', '[]');
											$users_json = $this->request->getData('users_json', '[]');
											$width = $this->request->getData('width');
											$height = $this->request->getData('height');
											$firebase_video_url = $this->request->getData('firebase_video_url');
											$lat = $this->request->getData('lat');
											$long = $this->request->getData('long');
											$story = $this->request->getData('story');
											$location_string = $this->request->getData('location_string');
											$location_name = $this->request->getData('location_name');
											$place_id = $this->request->getData('google_place_id');
											$tag_store_id = $this->request->getData('tag_store_id', 0);
											$thum = $this->request->getData('thum');
											$user_thumbnail = $this->request->getData('user_thumbnail');
											$default_thumbnail = $this->request->getData('default_thumbnail');
											$sound = $this->Sound->getDetails($sound_id);
											if (MEDIA_STORAGE === 's3') {
												$result_video = $this->aws->s3_video_upload($user_id, 'video', $sound, [], 0);
												$video_url_nudity_check = $result_video['video'];
												$video_url = strlen(CLOUDFRONT_URL) > 5 ? Utility::getCloudFrontUrl($result_video['video'], '/video') : $result_video['video'];
												$gif_url = strlen(CLOUDFRONT_URL) > 5 ? Utility::getCloudFrontUrl($result_video['gif'], '/gif') : $result_video['gif'];
												$thum_url = strlen(CLOUDFRONT_URL) > 5 ? Utility::getCloudFrontUrl($result_video['thum'], '/thum') : $result_video['thum'];
												$audio_url = strlen(CLOUDFRONT_URL) > 5 ? Utility::getCloudFrontUrl($result_video['audio'], '/audio') : $result_video['audio'];
												$thum_small_url = strlen(CLOUDFRONT_URL) > 5 ? Utility::getCloudFrontUrl($result_video['thum_small'], '/thum') : $result_video['thum_small'];
											} else {
												$result_video = Regular::local_video_upload($user_id, 'video', $sound, [], 0);
												$video_url = $result_video['video'];
												$gif_url = $result_video['gif'];
												$thum_url = $result_video['thum'];
												$audio_url = $result_video['audio'];
												$video_url_nudity_check = BASE_URL . $video_url;
												$thum_small_url = $thum_url;
											}
											$video_save = [
												'sound_id' => $sound_id,
												'gif' => $gif_url,
												'duration' => $ffmpeg->getDurationOfVideoFile($result_video['video']),
												'video' => $video_url,
												'thum' => $thum_url,
												'thum_small' => $thum_small_url,
												'description' => $description,
												'privacy_type' => $privacy_type,
												'allow_comments' => $allow_comments,
												'story' => $story,
												'user_id' => $user_id,
												'firebase_video_url' => $firebase_video_url,
												'width' => $width,
												'height' => $height,
												'user_thumbnail' => $user_thumbnail,
												'default_thumbnail' => $default_thumbnail,
												'lat' => $lat,
												'long' => $long,
												'tag_store_id' => $tag_store_id,
												'location_string' => $location_string,
												'created' => $created
											];
											if (empty($default_thumbnail)) {
												$video_save['user_thumbnail'] = $thum;
												$video_save['default_thumbnail'] = $thum;
											}
											if (!empty($thum)) {
												$video_save['thum'] = $thum;
											}
											if (!empty($place_id)) {
												$location = $this->Location->ifPlaceIDExist($place_id);
												if (!empty($location)) {
													$video_save['location_id'] = $location['Location']['id'];
												} else {
													$this->Location->save([
														'google_place_id' => $place_id,
														'name' => $location_name,
														'lat' => $lat,
														'long' => $long,
														'string' => $location_string,
														'created' => $created
													]);
													$video_save['location_id'] = $this->Location->id;
												}
											}
											if ($user_id < 1) {
												Message::EMPTYDATA();
												return;
											}
											if (!$this->Video->save($video_save)) {
												echo Message::DATASAVEERROR();
												return;
											}
											$video_id = $this->Video->id;
											if (!file_exists(ROOT . '/webroot/cron.txt') && strlen(DEEPENGIN_KEY) > 5) {
												Utility::checkNudity($video_url_nudity_check, $video_id);
												Utility::compressVideo($video_url, $video_id);
											}
											foreach (json_decode($hashtags_json, true) as $tag) {
												$name = strtolower($tag['name']);
												$exists = $this->Hashtag->ifExist($name);
												$hashtag_id = empty($exists) ? $this->Hashtag->save(['name' => $name]) && $this->Hashtag->id : $exists['Hashtag']['id'];
												$this->HashtagVideo->save(['hashtag_id' => $hashtag_id, 'video_id' => $video_id]);
												$this->Hashtag->clear();
											}
											foreach (json_decode($users_json, true) as $mention) {
												$tagged = $this->User->getUserDetailsFromID($mention['user_id']);
												$msg = $user['User']['username'] . ' has tagged you in a video';
												if (!empty($tagged['User']['device_token'])) {
													$notification = [
														'to' => $tagged['User']['device_token'],
														'notification' => ['title' => $msg, 'body' => '', 'badge' => '1', 'sound' => 'default', 'type' => 'video_tag', 'receiver_id' => $tagged['User']['id']],
														'data' => ['title' => $msg, 'body' => '', 'type' => 'video_tag', 'receiver_id' => $tagged['User']['id']]
													];
													$prefs = $this->PushNotification->getDetails($tagged['User']['id']);
													if (!empty($prefs) && $prefs['PushNotification']['video_updates'] > 0) {
														$this->firebase->sendPushNotification($notification);
													}
													$this->Notification->save([
														'sender_id' => $user_id,
														'receiver_id' => $tagged['User']['id'],
														'type' => 'video_tag',
														'video_id' => $video_id,
														'string' => $msg,
														'created' => $created
													]);
													Cache::delete('notifications_' . $tagged['User']['id']);
												}
											}
											foreach ($this->Follower->getUserFollowersWithoutLimit($user_id) as $follower) {
												$device_token = $follower['FollowerList']['device_token'];
												$msg = $user['User']['username'] . ' has posted a video';
												if (!empty($device_token)) {
													$notification = [
														'to' => $device_token,
														'notification' => ['title' => $msg, 'body' => '', 'badge' => '1', 'sound' => 'default', 'type' => 'video_new_post', 'video_id' => $video_id, 'receiver_id' => $follower['FollowerList']['id']],
														'data' => ['title' => $msg, 'body' => '', 'type' => 'video_new_post', 'video_id' => $video_id, 'receiver_id' => $follower['FollowerList']['id']]
													];
													$prefs = $this->PushNotification->getDetails($follower['FollowerList']['id']);
													if (!empty($prefs) && $prefs['PushNotification']['video_updates'] > 0) {
														$this->firebase->sendPushNotification($notification);
													}
													$this->Notification->save([
														'sender_id' => $user_id,
														'receiver_id' => $follower['FollowerList']['id'],
														'type' => 'video_updates',
														'video_id' => $video_id,
														'string' => $msg,
														'created' => $created
													]);
													Cache::delete('notifications_' . $follower['FollowerList']['id']);
												}
											}
											$details = $this->Video->getDetails($video_id);
											echo json_encode(['code' => 200, 'msg' => $details]);
										}
									}

									public function editVideo(): void
									{
										$this->loadModel('Video');
										$this->loadModel('Sound');
										$this->loadModel('Hashtag');
										$this->loadModel('HashtagVideo');
										$this->loadModel('User');
										$this->loadModel('Notification');
										$this->loadModel('Follower');
										$this->loadModel('PushNotification');
										$this->loadModel('VideoProduct');
										$this->loadModel('Location');
										if ($this->request->is('post')) {
											$user = $this->user_details;
											if (empty($user)) {
												Message::UNAUTHORIZED();
												return;
											}
											$user_id = $user['User']['id'];
											$video_id = $this->request->getData('video_id');
											$video_details = $this->Video->getOnlyVideoDetails($video_id);
											if (empty($video_details)) {
												Message::EMPTYDATA();
												return;
											}
											$description = $this->request->getData('description');
											$privacy_type = strtolower($this->request->getData('privacy_type'));
											$allow_comments = $this->request->getData('allow_comments');
											$user_thumbnail = $this->request->getData('user_thumbnail');
											$default_thumbnail = $this->request->getData('default_thumbnail');
											$hashtags_json = $this->request->getData('hashtags_json', '[]');
											$products_json = $this->request->getData('products_json', '[]');
											$location_name = $this->request->getData('location_name');
											$google_place_id = $this->request->getData('google_place_id');
											$lat = $this->request->getData('lat');
											$long = $this->request->getData('long');
											$location_string = $this->request->getData('location_string');
											$thum = $this->request->getData('thum');
											$video_save = [
												'description' => $description,
												'privacy_type' => $privacy_type,
												'allow_comments' => $allow_comments,
												'location_name' => $location_name,
												'user_id' => $user_id,
												'user_thumbnail' => $user_thumbnail,
												'default_thumbail' => $default_thumbnail,
												'lat' => $lat,
												'long' => $long,
												'location_string' => $location_string
											];
											if (!empty($thum)) {
												$video_save['thum'] = $thum;
											}
											if (!empty($google_place_id)) {
												$location = $this->Location->ifPlaceIDExist($google_place_id);
												if (!empty($location)) {
													$video_save['location_id'] = $location['Location']['id'];
												} else {
													$this->Location->save([
														'google_place_id' => $google_place_id,
														'name' => $location_name,
														'lat' => $lat,
														'long' => $long,
														'string' => $location_string
													]);
													$video_save['location_id'] = $this->Location->id;
												}
											}
											if (!empty(json_decode($products_json, true))) {
												$this->VideoProduct->deleteDataAgainstVideo($video_id);
												foreach (json_decode($products_json, true) as $prod) {
													if (!empty($prod['product_id'])) {
														$this->VideoProduct->save(['product_id' => $prod['product_id'], 'video_id' => $video_id]);
													}
												}
											}
											$this->Video->id = $video_id;
											if (!$this->Video->save($video_save)) {
												echo Message::DATASAVEERROR();
												return;
											}
											$this->HashtagVideo->deleteDataAgainstVideo($video_id);
											foreach (json_decode($hashtags_json, true) as $tag) {
												$name = strtolower($tag['name']);
												$exists = $this->Hashtag->ifExist($name);
												$hashtag_id = empty($exists) ? $this->Hashtag->save(['name' => $name]) && $this->Hashtag->id : $exists['Hashtag']['id'];
												$this->HashtagVideo->save(['hashtag_id' => $hashtag_id, 'video_id' => $video_id]);
												$this->Hashtag->clear();
											}
											Cache::delete('video_details_' . $video_id);
											$details = $this->Video->getDetails($video_id);
											echo json_encode(['code' => 200, 'msg' => $details]);
										}
									}

									public function sendPushToFollowers(int $user_id, string $type, array $data): void
									{
										$this->loadModel('Follower');
										$this->loadModel('PushNotification');
										$this->loadModel('Notification');
										$created = date('Y-m-d H:i:s');
										$followers = $this->Follower->getUserFollowersWithoutLimit($user_id);
										if (!empty($followers)) {
											foreach ($followers as $follower) {
												$fid = $follower['FollowerList']['id'];
												$token = $follower['FollowerList']['device_token'];
												$msg = $data['msg'] ?? '';
												if (!empty($token)) {
													$notification = [
														'to' => $token,
														'notification' => [
															'title' => $msg,
															'body' => '',
															'badge' => '1',
															'sound' => 'default',
															'type' => $type,
															'video_id' => $data['video_id'] ?? null,
															'receiver_id' => $fid
														],
														'data' => [
															'title' => $msg,
															'body' => '',
															'type' => $type,
															'video_id' => $data['video_id'] ?? null,
															'receiver_id' => $fid
														]
													];
													$prefs = $this->PushNotification->getDetails($fid);
													if (!empty($prefs) && $prefs['PushNotification']['video_updates'] > 0) {
														$this->firebase->sendPushNotification($notification);
													}
													$this->Notification->save([
														'sender_id' => $this->user_details['User']['id'],
														'receiver_id' => $fid,
														'type' => $type,
														'video_id' => $data['video_id'] ?? null,
														'string' => $msg,
														'created' => $created
													]);
													Cache::delete('notifications_' . $fid);
												}
											}
										}
									}
    
										public function showFriends(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Follower');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$friends = [];
											$userId = $data['user_id'] ?? null;
											$startingPoint = $data['starting_point'] ?? 0;
											$followers = $this->Follower->getUserFollowers($userId, $startingPoint);
											foreach ($followers as $key => $val) {
												$followerBackDetails = $this->Follower->ifFollowing($userId, $val['FollowerList']['id']);
												if (!empty($followerBackDetails)) {
													$friends[$key]['Friends'] = $val['FollowerList'];
												}
											}
											$output = !empty($friends) ? ['code' => 200, 'msg' => $friends] : ['code' => 204, 'msg' => []];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showStoreUsers(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('User');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$startingPoint = $data['starting_point'] ?? 0;
											$details = $this->User->getBusinessUsers($startingPoint);
											$output = !empty($details) ? ['code' => 200, 'msg' => $details] : ['code' => 204, 'msg' => []];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showFriendsStories(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Video');
											$this->loadModel('Follower');
											$this->loadModel('User');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$userId = $data['user_id'] ?? null;
											$startingPoint = $data['starting_point'] ?? 0;
											$oneDayBefore = date('Y-m-d H:i:s', strtotime('-1 days'));
											$followers = $this->Follower->getUserFollowing($userId, $startingPoint);
											$friendStories = [];
											$i = 0;
											$userDetails = $this->User->getOnlyUserDetailsFromID($userId);
											$userStory = $this->Video->getUserStory($userId, $oneDayBefore);
											if (!empty($userStory)) {
												$friendStories['User'][$i] = $userDetails['User'];
												$friendStories['User'][$i]['Video'] = $userStory;
												$i++;
											}
											foreach ($followers as $follow) {
												$receiverId = $follow['FollowingList']['id'];
												$ifFollowing = $this->Follower->ifFollowing($receiverId, $userId);
												if (!empty($ifFollowing)) {
													$friendStories['User'][$i] = $ifFollowing['FollowerList'];
													$friendStories['User'][$i]['Video'] = $this->Video->getUserStory($receiverId, $oneDayBefore);
													$i++;
												}
											}
											$output = !empty($friendStories) ? ['code' => 200, 'msg' => $friendStories] : ['code' => 204, 'msg' => []];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showFollowingVideos(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Video');
											$this->loadModel('VideoComment');
											$this->loadModel('VideoLike');
											$this->loadModel('VideoFavourite');
											$this->loadModel('Follower');
											$this->loadModel('RepostVideo');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$startingPoint = $data['starting_point'] ?? 0;
											$userDetails = $this->user_details;
											if (empty($userDetails)) {
												$output = ['code' => 201, 'msg' => 'Please login first'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											$userId = $userDetails['User']['id'];
											$followingUsers = $this->Follower->getUserFollowingWithoutLimit($userId);
											$ids = [];
											foreach ($followingUsers as $following) {
												$ids[] = $following['FollowingList']['id'];
											}
											if (empty($ids)) {
												$output = ['code' => 201, 'msg' => 'you are not following anyone yet'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											$videos = $this->Video->getFollowingVideos($userId, $startingPoint, $ids);
											if (empty($videos)) {
												$output = ['code' => 204, 'msg' => []];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											foreach ($videos as $key => $video) {
												$this->customizeVideoResponse($videos[$key], $userId);
												unset($videos[$key]['VideoProduct']);
											}
											$output = ['code' => 200, 'msg' => $videos];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showVideosForCompressionAndQualityCron(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Video');
											$this->request->allowMethod(['post']);
											$details = $this->Video->getVideosWhichAreNotCompressedAndNotQualityChecked();
											$output = !empty($details) ? ['code' => 200, 'msg' => $details] : ['code' => 204, 'msg' => []];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showCountries(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Country');
											$this->request->allowMethod(['post']);
											$countries = $this->Country->getAll();
											$output = ['code' => 200, 'msg' => $countries];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function editProfile(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('User');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$userDetails = $this->user_details;
											if (empty($userDetails)) {
												throw new \Cake\Http\Exception\UnauthorizedException();
											}
											$userId = $userDetails['User']['id'];
											Cache::delete('user_details_' . $userId);
											if (isset($data['user_id'])) {
												$output = ['code' => 201, 'msg' => 'Something went wrong. Please try again'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											if (!empty($data['email'])) {
												$emailExist = $this->User->editIsEmailAlreadyExist($data['email'], $userId);
												if ($emailExist > 0) {
													$output = ['code' => 201, 'msg' => 'This email has already been registered'];
													return $this->response->withType('application/json')->withStringBody(json_encode($output));
												}
											}
											$user = [];
											foreach (['first_name','profile_gif','profile_pic','profile_view','last_name','bio','website','phone','private','facebook','instagram','youtube','twitter','username'] as $field) {
												if (isset($data[$field])) {
													$user[$field] = $data[$field];
													if ($field === 'profile_pic' && !empty($userDetails['User']['profile_pic'])) {
														$this->aws->deleteObjectS3($userDetails['User']['profile_pic']);
													}
													if ($field === 'username') {
														$usernameExist = $this->User->editIsUsernameAlreadyExist($data['username'], $userId);
														if ($usernameExist > 0) {
															$output = ['code' => 201, 'msg' => 'username already exist'];
															return $this->response->withType('application/json')->withStringBody(json_encode($output));
														}
													}
												}
											}
											if (empty($user)) {
												$output = ['code' => 201, 'msg' => 'nothing to update'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											$this->User->id = $userId;
											$this->User->save($user);
											$this->User->clear();
											$userDetails = $this->User->getUserDetailsFromID($userId);
											$this->populateUserStats($userDetails, $userId, $userId);
											$output = ['code' => 200, 'msg' => $userDetails];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function logout(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('User');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$userId = $data['user_id'] ?? null;
											$this->User->id = $userId;
											$this->User->save(['device_token' => '', 'auth_token' => '']);
											$userDetails = $this->User->getUserDetailsFromID($userId);
											$output = !empty($userDetails) ? ['code' => 200, 'msg' => $userDetails] : ['code' => 204, 'msg' => []];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showAudiencesReach(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Audience');
											$this->loadModel('User');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$minAge = $data['min_age'] ?? 0;
											$maxAge = $data['max_age'] ?? 0;
											$gender = strtolower($data['gender'] ?? 'all');
											$locations = $data['locations'] ?? [];
											$totalReach = 0;
											if ($minAge < 1 && $maxAge < 1) {
												foreach ($locations as $location) {
													$aud = $this->User->totalAudienceAgainstCityID($location['city_id']);
													if (empty($aud)) {
														$aud = $this->User->totalAudienceAgainstStateID($location['state_id']);
														if (empty($aud)) {
															$aud = $this->User->totalAudienceAgainstCountryID($location['country_id']);
														}
													}
													$totalReach += $aud;
												}
											} elseif ($gender === 'all') {
												foreach ($locations as $location) {
													$aud = $this->User->totalAudienceWithoutGenderAndCity($minAge, $maxAge, $location['city_id']);
													if (empty($aud)) {
														$aud = $this->User->totalAudienceWithoutGenderAndState($minAge, $maxAge, $location['state_id']);
														if (empty($aud)) {
															$aud = $this->User->totalAudienceWithoutGenderAndCountry($minAge, $maxAge, $location['country_id']);
														}
													}
													$totalReach += $aud[0][0]['total_audience'];
												}
											} else {
												foreach ($locations as $location) {
													$aud = $this->User->totalAudienceAgainstGenderAndCity($minAge, $maxAge, $gender, $location['city_id']);
													if (empty($aud)) {
														$aud = $this->User->totalAudienceAgainstGenderAndState($minAge, $maxAge, $gender, $location['state_id']);
														if (empty($aud)) {
															$aud = $this->User->totalAudienceAgainstGenderAndCountry($minAge, $maxAge, $gender, $location['country_id']);
														}
													}
													$totalReach += $aud[0][0]['total_audience'];
												}
											}
											$output = ['code' => 200, 'msg' => $totalReach];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showLocations(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('User');
											$this->loadModel('Country');
											$this->loadModel('City');
											$this->loadModel('State');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$keyword = $data['keyword'] ?? '';
											$countries = $this->Country->getCountriesAgainstKeyword($keyword);
											$location = [];
											foreach ($countries as $country) {
												$id = $country[0]['id'];
												$name = $country[0]['name'];
												$type = $country[0]['type'];
												if ($type === 'country') {
													$location[] = ['country_id' => $id, 'state_id' => 0, 'city_id' => 0, 'name' => $name];
												} elseif ($type === 'state') {
													$details = $this->State->getDetails($id);
													$location[] = ['country_id' => $details['Country']['id'], 'state_id' => $id, 'city_id' => 0, 'name' => $name . ',' . $details['Country']['name']];
												} elseif ($type === 'city') {
													$details = $this->City->getDetails($id);
													$location[] = ['country_id' => $details['City']['country_id'], 'state_id' => $details['State']['id'], 'city_id' => $id, 'name' => $name . ',' . $details['Country']['name']];
												}
											}
											$output = !empty($location) ? ['code' => 200, 'msg' => $location] : ['code' => 204, 'msg' => []];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function addAudience(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Audience');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$audience = [
												'name' => $data['name'],
												'user_id' => $data['user_id'],
												'min_age' => $data['min_age'],
												'max_age' => $data['max_age'],
												'gender' => strtolower($data['gender']),
												'created' => date('Y-m-d H:i:s')
											];
											$this->Audience->save($audience);
											$details = $this->Audience->getDetails($this->Audience->id);
											$output = ['code' => 200, 'msg' => $details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function addDefaultAudience(): int
										{
											$this->loadModel('Audience');
											$audience = [
												'name' => 'default',
												'user_id' => 0,
												'min_age' => 18,
												'max_age' => 100,
												'gender' => 'all',
												'created' => date('Y-m-d H:i:s')
											];
											$this->Audience->save($audience);
											return $this->Audience->id;
										}
										public function deleteAudience(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Audience');
											$this->loadModel('Promotion');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$id = $data['id'];
											$created = date('Y-m-d H:i:s');
											$details = $this->Promotion->getActivePromotionAudience($id, $created);
											if (empty($details)) {
												$this->Audience->delete($id);
												$output = ['code' => 200, 'msg' => 'deleted'];
											} else {
												$output = ['code' => 201, 'msg' => 'You cannot delete an audience since your ad is active against this audience.'];
											}
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showAudiences(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Audience');
											$this->loadModel('User');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$audiences = $this->Audience->getUserAudiences($data['user_id']);
											$output = ['code' => 200, 'msg' => $audiences];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function sendGift(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Users');
											$this->loadModel('PushNotifications');
											$request = $this->getRequest();
											$response = $this->getResponse();
											$request->allowMethod(['post']);
											$data = $request->getData();

											$locator = \Cake\ORM\TableRegistry::getTableLocator();
											if (!$locator->exists('GiftSends')) {
												$output = ['code' => 201, 'msg' => ''];
												return $response->withType('application/json')->withStringBody(json_encode($output));
											}

											$this->loadModel('GiftSends');
											$this->loadModel('Gifts');

											try {
												$gift = $this->Gifts->get($data['gift_id']);
											} catch (\Cake\Datasource\Exception\RecordNotFoundException $e) {
												$output = ['code' => 204, 'msg' => []];
												return $response->withType('application/json')->withStringBody(json_encode($output));
											}

											$giftData = [
												'image'       => $gift->image,
												'coin'        => $gift->coin * $data['gift_count'],
												'gift_id'     => $gift->id,
												'title'       => $gift->title,
												'sender_id'   => $data['sender_id'],
												'receiver_id' => $data['receiver_id'],
											];

											$sender = $this->Users->getUserDetailsFromID($data['sender_id']);
											if ($this->walletTotal($sender->id)['total'] < $giftData['coin']) {
												$output = ['code' => 201, 'msg' => 'You do not have enough coins to send gift'];
												return $response->withType('application/json')->withStringBody(json_encode($output));
											}

											$entity = $this->GiftSends->newEntity($giftData);
											$this->GiftSends->save($entity);

											$receiver = $this->Users->getUserDetailsFromID($data['receiver_id']);
											$notification = [
												'to'           => $receiver->device_token,
												'notification' => [
													'title'  => 'You have received a gift',
													'body'   => $giftData['title'] . ' worth ' . $giftData['coin'] . ' coins',
													'user_id'=> $sender->id,
													'image'  => $sender->profile_pic,
													'name'   => $sender->username,
													'badge'  => '1',
													'sound'  => 'default',
													'icon'   => '',
													'type'   => 'gift',
												],
												'data'         => [
													'title'  => 'You have received a gift',
													'body'   => $giftData['title'] . ' worth ' . $giftData['coin'] . ' coins',
													'user_id'=> $sender->id,
													'image'  => $sender->profile_pic,
													'name'   => $sender->username,
													'badge'  => '1',
													'sound'  => 'default',
													'icon'   => '',
													'type'   => 'gift',
												],
											];

											$existing = $this->PushNotifications->getDetails($data['receiver_id']);
											if (!empty($existing['direct_messages'])) {
												$this->firebase->sendPushNotification($notification);
											}

											$sender = $this->Users->getUserDetailsFromID($data['sender_id']);
											$sender->wallet = $this->walletTotal($data['sender_id'])['total'];
											$output = ['code' => 200, 'msg' => $sender];

											return $response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function withdrawRequest(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('User');
											$this->loadModel('Payout');
											$this->loadModel('WithdrawRequest');
											$this->loadModel('VerificationRequest');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$verificationDetails = $this->VerificationRequest->ifRequestAlreadySubmitted($data['user_id']);
											if (!empty($verificationDetails)) {
												$output = ['code' => 201, 'msg' => 'Your account is under process for verification. You can withdraw your balance once your account has been verified'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											$payoutDetails = $this->Payout->ifExist($data['user_id']);
											if (empty($payoutDetails)) {
												$output = ['code' => 201, 'msg' => 'Please add first payout'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											$details = $this->WithdrawRequest->getUserPendingWithdrawRequest($data['user_id']);
											if (!empty($details)) {
												$output = ['code' => 201, 'msg' => 'You have already requested a payout.'];
												return $this->response->withType('application/json')->withStringBody(json_encode($output));
											}
											$this->WithdrawRequest->save([
												'user_id' => $data['user_id'],
												'amount' => $data['amount'],
												'coin' => $data['coin'] ?? null,
												'created' => date('Y-m-d H:i:s'),
												'payout_method' => $data['payout_method'],
												'payout_value' => $data['payout_value']
											]);
											$details = $this->WithdrawRequest->getDetails($this->WithdrawRequest->id);
											$details['User']['wallet'] = $this->walletTotal($data['user_id'])['total'];
											$output = ['code' => 200, 'msg' => $details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function addStory(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Story');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$ffmpeg = new Ffmpeg();
											$videoDuration = $ffmpeg->getDurationOfVideoFile($data['attachment']);
											$story = [
												'type' => $data['type'],
												'user_id' => $data['user_id'],
												'attachment' => $data['attachment'],
												'created' => $data['datetime'],
												'duration' => $videoDuration
											];
											$this->Story->save($story);
											$details = $this->Story->getDetails($this->Story->id);
											$output = ['code' => 200, 'msg' => $details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function showStory(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Story');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$date = $data['date'];
											$storyOthers = $this->Story->getOtherStories($date);
											$myStory = isset($data['user_id']) ? $this->Story->getUserStories($data['user_id'], $date) : [];
											$output = ['code' => 200, 'msg' => ['mystory' => $myStory, 'other' => $storyOthers]];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function deleteStory(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Story');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$this->Story->delete($data['id']);
											$output = ['code' => 200, 'msg' => 'success'];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
										public function addGroup(): \Psr\Http\Message\ResponseInterface
										{
											$this->loadModel('Group');
											$this->request->allowMethod(['post']);
											$data = $this->request->getData();
											$postData = ['title' => $data['title'], 'user_id' => $data['user_id'], 'created' => date('Y-m-d H:i:s')];
											if (isset($data['id'])) {
												$groupDetails = $this->Group->getDetails($data['id']);
												if (!empty($data['image'])) {
													@unlink($groupDetails['Group']['image']);
													$postData['image'] = Utility::uploadFileintoFolderDir($data['image'], UPLOADS_FOLDER_URI);
												}
												$this->Group->save($postData + ['id' => $data['id']]);
												$details = $this->Group->getDetails($data['id']);
											} else {
												if (!empty($data['image'])) {
													$postData['image'] = Utility::uploadFileintoFolderDir($data['image'], UPLOADS_FOLDER_URI);
												}
												$this->Group->save($postData);
												$details = $this->Group->getDetails($this->Group->id);
											}
											$output = ['code' => 200, 'msg' => $details];
											return $this->response->withType('application/json')->withStringBody(json_encode($output));
										}
    
											public function pinComment(): void {
												$this->loadModel('Video');
												$this->loadModel('VideoComment');
												if ($this->request->is('post')) {
													$json = $this->request->getInput();
													$data = json_decode($json, true);
													$required_params = ['video_id', 'pin_comment_id'];
													$this->verifyParameters($json, $data, $required_params);
													$pin = $data['pin'] ?? 0;
													$video_id = $data['video_id'];
													$pin_comment_id = $data['pin_comment_id'];
													$comment_details = $this->VideoComment->getDetails($pin_comment_id);
													if ($this->count_($comment_details) > 1) {
														$this->VideoComment->updatePin($video_id);
														$this->VideoComment->id = $pin_comment_id;
														$this->VideoComment->saveField('pin', $pin);
														$this->VideoComment->clear();
														$details = $this->Video->getDetails($video_id);
														echo json_encode(['code' => 200, 'msg' => $details]);
														die();
													}
													Message::EMPTYDATA();
													die();
												}
											}

											public function inviteUserToGroup(): void {
												$this->loadModel('Notification');
												$this->loadModel('Group');
												if ($this->request->is('post')) {
													$json = $this->request->getInput();
													$data = json_decode($json, true);
													$created = date('Y-m-d H:i:s');
													$group_details = $this->Group->getDetails($data['group_id']);
													$receiver_details = $this->User->getUserDetailsFromID($data['user_id']);
													if ($this->count_($group_details) > 0) {
														$msg = "You have been invite in " . $group_details['Group']['title'] . " group";
														$notification_data = [
															'sender_id'   => $group_details['Group']['user_id'],
															'receiver_id' => $data['user_id'],
															'group_id'    => $group_details['Group']['id'],
															'type'        => 'group_invite',
															'string'      => $msg,
															'created'     => $created
														];
														$this->Notification->save($notification_data);
														Cache::delete('notifications_' . $data['user_id']);
														$id = $this->Notification->id;
														$notification = [
															'to'            => $receiver_details['User']['device_token'],
															'notification'  => ['title'=>'group invitation','body'=>$msg,'badge'=>'1','sound'=>'default','icon'=>'','type'=>'live','user_id'=>$data['user_id']],
															'data'          => ['title'=>'group invitation','body'=>$msg,'icon'=>'','badge'=>'1','sound'=>'default','type'=>'live','user_id'=>$data['user_id']]
														];
														$this->firebase->sendPushNotification($notification);
														$details = $this->Notification->getDetails($id);
														echo json_encode(['code'=>200,'msg'=>$details]);
														die();
													}
												}
											}

											public function pinVideo(): void {
												$this->loadModel('Video');
												if ($this->request->is('post')) {
													$json = $this->request->getInput();
													$data = json_decode($json, true);
													$details = $this->Video->getDetails($data['video_id']);
													if ($this->count_($details) > 0) {
														$this->Video->id = $data['video_id'];
														$this->Video->saveField('pin', $data['pin']);
														$details = $this->Video->getDetails($data['video_id']);
														echo json_encode(['code'=>200,'msg'=>$details]);
														die();
													}
													Message::EMPTYDATA();
													die();
												}
											}

											public function acceptGroupInvite(): void {
												$this->loadModel('Notification');
												if ($this->request->is('post')) {
													$json = $this->request->getInput();
													$data = json_decode($json, true);
													$details = $this->Notification->getDetails($data['id']);
													if ($this->count_($details) > 0) {
														$this->Notification->id = $data['id'];
														$this->Notification->saveField('status', $data['status']);
														$details = $this->Notification->getDetails($data['id']);
														echo json_encode(['code'=>200,'msg'=>$details]);
														die();
													}
												}
											}

											public function addPayout(): void {
												$this->loadModel('Payout');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													if ($this->count_($this->user_details) === 0) {
														Message::UNAUTHORIZED();
														die();
													}
													$user_id = $this->user_details['User']['id'];
													$post_data = [
														'user_id' => $user_id,
														'type'    => $data['type'],
														'value'   => $data['value'],
														'created' => date('Y-m-d H:i:s')
													];
													$existing = $this->Payout->ifExist($user_id);
													if ($this->count_($existing) > 0) {
														$this->Payout->id = $existing['Payout']['id'];
														$this->Payout->save($post_data);
														$id = $this->Payout->id;
													} else {
														$this->Payout->save($post_data);
														$id = $this->Payout->id;
													}
													$details = $this->Payout->getDetails($id);
													echo json_encode(['code'=>200,'msg'=>$details]);
													die();
												}
											}

											public function addProductFavourite(): void {
												$this->loadModel('ProductFavourite');
												$this->loadModel('Product');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													$user_id = $this->count_($this->user_details) > 0 ? $this->user_details['User']['id'] : 0;
													$details = $this->Product->getDetails($data['product_id']);
													if ($this->count_($details) > 0) {
														$fav = $this->ProductFavourite->ifProductFavourite($user_id, $data['product_id']);
														if ($this->count_($fav) < 1) {
															$this->ProductFavourite->save(['product_id'=>$data['product_id'],'created'=>date('Y-m-d H:i:s')]);
															$msg = $this->ProductFavourite->getDetails($this->ProductFavourite->id);
														} else {
															$this->ProductFavourite->deleteFavourite($user_id, $data['product_id']);
															$msg = 'deleted';
														}
														echo json_encode(['code'=>200,'msg'=>$msg]);
													} else {
														echo json_encode(['code'=>201,'msg'=>'Product do not exist']);
													}
													die();
												}
											}

											public function count_(mixed $array): int {
												return is_array($array) ? count($array) : 0;
											}

											public function addProductImage(): void {
												$this->loadModel('ProductImage');
												$this->loadModel('Product');
												if ($this->request->is('post')) {
													$product_id = $this->request->getData('product_id');
													$details = $this->Product->getDetails($product_id);
													if ($this->count_($details) > 0) {
														$file = $this->request->getData('file');
														$upload = Utility::uploadAMultipartFileIntoFolder($file['tmp_name'],'png',UPLOADS_FOLDER_URI);
														if (MEDIA_STORAGE==='s3') {
															$upload = $this->aws->fileUploadToS3($upload,'png',UPLOADS_FOLDER_URI);
														}
														if ($upload['code']===200) {
															$path = str_replace('tictic-video/','',$upload['msg']);
															$cdn = CLOUDFRONT_URL . parse_url($path, PHP_URL_PATH);
															$this->ProductImage->save(['product_id'=>$product_id,'image'=>$cdn,'thum'=>$cdn,'created'=>date('Y-m-d H:i:s')]);
															$msg = $this->ProductImage->getDetails($this->ProductImage->id);
															echo json_encode(['code'=>200,'msg'=>$msg]);
															die();
														}
														echo json_encode(['code'=>201,'msg'=>'Something went wrong in saving the data']);
														die();
													}
													Message::EMPTYDATA();
													die();
												}
											}

											public function addProductImages(): void {
												$this->loadModel('Product');
												$this->loadModel('ProductImage');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													$product_id = $data['product_id'];
													$details = $this->Product->getDetails($product_id);
													if ($this->count_($details) > 0 && !empty($data['images'])) {
														$existing = $this->ProductImage->getProductImageAgainstProductID($product_id);
														foreach ($existing as $img) {
															if (strlen($img['ProductImage']['image'])>5) unlink($img['ProductImage']['image']);
														}
														$this->ProductImage->deleteAll(['product_id'=>$product_id], false);
														$toSave = [];
														foreach ($data['images'] as $img) {
															$path = Utility::uploadFileintoFolder($product_id,$img['image'],UPLOADS_FOLDER_URI);
															$toSave[] = ['product_id'=>$product_id,'image'=>$path,'created'=>date('Y-m-d H:i:s')];
														}
														$this->ProductImage->saveAll($toSave);
													}
													$msg = $this->Product->getDetails($product_id);
													echo json_encode(['code'=>200,'msg'=>$msg]);
													die();
												}
											}

											public function deleteProductAttribute(): void {
												$this->loadModel('ProductAttribute');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													$details = $this->ProductAttribute->getDetails($data['id']);
													if ($this->count_($details) > 0) {
														$this->ProductAttribute->delete($data['id'], true);
														echo json_encode(['code'=>200,'msg'=>'deleted']);
														die();
													}
													Message::ALREADYDELETED();
													die();
												}
											}

											public function deleteProduct(): void {
												$this->loadModel('Product');
												$this->loadModel('ProductImage');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													$details = $this->Product->getDetails($data['product_id']);
													if ($this->count_($details) > 0) {
														$images = $this->ProductImage->getProductImageAgainstProductID($data['product_id']);
														foreach ($images as $img) {
															if (strlen($img['ProductImage']['image'])>5) unlink($img['ProductImage']['image']);
														}
														$this->Product->delete($data['product_id'], true);
														echo json_encode(['code'=>200,'msg'=>'deleted']);
														die();
													}
												}
											}

											public function deleteProductSingleImage(): void {
												$this->loadModel('ProductImage');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													$details = $this->ProductImage->getDetails($data['product_image_id']);
													if ($this->count_($details) > 0) {
														if (strlen($details['ProductImage']['image'])>5) unlink($details['ProductImage']['image']);
														$this->ProductImage->delete($data['product_image_id']);
														echo json_encode(['code'=>200,'msg'=>'deleted']);
														die();
													}
												}
											}

											public function addPromotion(): void {
												$this->loadModel('Promotion');
												if ($this->request->is('post')) {
													$data = json_decode($this->request->getInput(), true);
													$audience = $data['audience_id'] < 1 ? $this->addDefaultAudience() : $data['audience_id'];
													$promotion = [
														'video_id'       => $data['video_id'],
														'user_id'        => $data['user_id'],
														'destination'    => $data['destination'],
														'coin'           => $data['coin'],
														'audience_id'    => $audience,
														'total_reach'    => $data['total_reach'],
														'start_datetime' => $data['start_datetime'],
														'end_datetime'   => $data['end_datetime'],
														'active'         => 0,
														'created'        => date('Y-m-d H:i:s')
													];
													if ($promotion['destination']==='website') {
														$promotion['website_url']  = $data['website_url'];
														$promotion['action_button']= $data['action_button'];
													}
													$this->Promotion->save($promotion);
													$details = $this->Promotion->getDetails($this->Promotion->id);
													$details['User']['wallet'] = $this->walletTotal($promotion['user_id'])['total'];
													echo json_encode(['code'=>200,'msg'=>$details]);
													die();
												}
											}
    
												public function walletTotal(int $userId): array
												{
													$withdrawRequestsTable       = $this->getTableLocator()->get('WithdrawRequests');
													$promotionsTable             = $this->getTableLocator()->get('Promotions');
													$liveStreamingsTable         = $this->getTableLocator()->get('LiveStreamings');
													$liveStreamingWatchesTable   = $this->getTableLocator()->get('LiveStreamingWatches');
													$referralUsedsTable          = $this->getTableLocator()->get('ReferralUseds');
													$settingsTable               = $this->getTableLocator()->get('Settings');
													$ordersTable                 = $this->getTableLocator()->get('Orders');
													$dailyCheckinsTable          = $this->getTableLocator()->get('DailyCheckins');
													$purchaseCoinsTable          = $this->getTableLocator()->get('PurchaseCoins');
													$giftSendsTable              = $this->getTableLocator()->get('GiftSends');

													$olderThanDate = \Cake\I18n\FrozenTime::now()->subDays(15)->format('Y-m-d H:i:s');

													$coinsDailyCheckin            = $dailyCheckinsTable->countCoins($userId)[0]['total_coin']            ?? 0;
													$withdrawRequestDetail        = $withdrawRequestsTable->getTotalCoins($userId)[0]['total_coin']      ?? 0;
													$withdrawRequestAmountDetail  = $withdrawRequestsTable->getTotalAmountWithdrawal($userId)[0]['total_amount'] ?? 0;
													$purchaseAmountTotal          = $purchaseCoinsTable->totalAmountPurchase($userId)[0]['total_amount']      ?? 0;
													$giftSend                     = $giftSendsTable->countGiftSendByUser($userId)[0]['total_amount']       ?? 0;
													$giftReceive                  = $giftSendsTable->countGiftReceiveByUser($userId)[0]['total_amount']    ?? 0;
													$userPlaceOrderCoins          = $ordersTable->getTotalCoins($userId)[0]['total_amount']               ?? 0;
													$sellerPendingCoins           = $ordersTable->getPendingSellerBalance($userId, $olderThanDate)[0]['total_amount'] ?? 0;
													$referralCount                = $referralUsedsTable->countReferralUsedByOthers($userId);
													$referralValue                = $settingsTable->checkDuplicate('referral_coin')['Setting']['value']    ?? 0;
													$referralEarn                 = $referralCount * $referralValue;
													$promotionCoin                = $promotionsTable->countPromotionCoin($userId)[0]['total_amount']      ?? 0;
													$liveStreamEarnedCoin         = $liveStreamingsTable->countCoinsEarnedByUser($userId)[0]['total_amount'] ?? 0;
													$liveStreamWatchCoin          = $liveStreamingWatchesTable->countCoinsSpendByUser($userId)[0]['total_amount']?? 0;
													$commissionEarned             = 0;
													$sellerBalance                = 0;

													$earnedMoney = $giftReceive
																 + $coinsDailyCheckin
																 + $purchaseAmountTotal
																 + $liveStreamEarnedCoin
																 + $referralEarn
																 + $sellerBalance / 100
																 + $commissionEarned / 100;

													$totalMoney  = $earnedMoney
																 - $giftSend
																 - $promotionCoin
																 - $liveStreamWatchCoin
																 - $withdrawRequestDetail
																 - $userPlaceOrderCoins;

													return [
														'gifts_receive'         => (string)($giftReceive + $liveStreamEarnedCoin),
														'gifts_send'            => (string)$giftSend,
														'total'                 => (string)$totalMoney,
														'comission_earned'      => (string)($commissionEarned / 100),
														'total_balance_usd'     => (string)( $commissionEarned / 100 + $sellerBalance / 100 - $withdrawRequestAmountDetail ),
														'seller_pending_balance'=> (string)$sellerPendingCoins
													];
												}

												public function destinationTap(): void
												{
													$this->loadModel('Promotion');
													if ($this->getRequest()->is('post')) {
														$data = $this->getRequest()->getData();
														$promotion_id = $data['promotion_id'];
														$details = $this->Promotion->getDetails($promotion_id);
														if (count($details) > 0) {
															$destination_tap = $details['Promotion']['destination_tap'];
															$this->Promotion->id = $promotion_id;
															$this->Promotion->saveField('destination_tap', $destination_tap + 1);
															$details = $this->Promotion->getDetails($promotion_id);
															$output = ['code' => 200, 'msg' => $details];
															echo json_encode($output);
															die();
														} else {
															Message::EMPTYDATA();
															die();
														}
													}
												}

												public function addDeviceData(): void
												{
													$this->loadModel('User');
													$this->loadModel('Country');
													$this->loadModel('State');
													$this->loadModel('City');
													$this->loadModel('PushNotification');
													$this->loadModel('PrivacySetting');
													if ($this->getRequest()->is('post')) {
														$data = $this->getRequest()->getData();
														$user = [
															'device_token' => $data['device_token'],
															'ip' => $data['ip'],
															'device' => $data['device'],
															'version' => $data['version']
														];
														$created = date('Y-m-d H:i:s');
														if (isset($data['lat'])) {
															$user['lat'] = $data['lat'];
															$user['long'] = $data['long'];
															$location_details = Utility::getCountryCityProvinceFromLatLong($data['lat'], $data['long']);
															if (count($location_details) > 0 && strlen($location_details['location_string']) > 2) {
																$user['country'] = $location_details['country'];
															}
														}
														$user_id = $data['user_id'];
														$userDetails = $this->User->getUserDetailsFromID($user_id);
														if (count($userDetails) > 0) {
															$this->User->id = $user_id;
															$this->User->save($user);
															if (count($this->PushNotification->getDetails($user_id)) < 1) {
																$this->PushNotification->save([
																	'likes' => 1, 'comments' => 1, 'new_followers' => 1,
																	'mentions' => 1, 'direct_messages' => 1,
																	'video_updates' => 1, 'id' => $user_id
																]);
															}
															if (count($this->PrivacySetting->getDetails($user_id)) < 1) {
																$this->PrivacySetting->save([
																	'videos_download' => 1, 'videos_repost' => 1,
																	'direct_message' => 'everyone', 'duet' => 'everyone',
																	'liked_videos' => 'me', 'video_comment' => 'everyone',
																	'id' => $user_id
																]);
															}
															$userDetails = $this->User->getUserDetailsFromID($user_id);
															echo json_encode(['code' => 200, 'msg' => $userDetails]);
														} else {
															Message::EMPTYDATA();
															die();
														}
													}
												}

												public function addPhoneDeviceData($device_token, $ip, $device, $version, $user_id)
												{
													$this->loadModel('User');
													$this->loadModel('Country');
													$this->loadModel('State');
													$this->loadModel('City');
													$this->loadModel('PushNotification');
													$this->loadModel('PrivacySetting');
													$user = [
														'device_token' => $device_token,
														'ip' => $ip,
														'device' => $device,
														'version' => $version
													];
													$ipdat = Utility::getLocationFromIP($ip);
													$state_id = 0;
													$country_id = 0;
													$city_id = 0;
													if (count($ipdat) > 0) {
														$state_name = strtolower($ipdat['geoplugin_region']);
														$state_short_name = strtolower($ipdat['geoplugin_regionCode']);
														$user['city'] = strtolower($ipdat['geoplugin_city']);
														$user['country'] = strtolower($ipdat['geoplugin_countryName']);
														$short_country_name = strtolower($ipdat['geoplugin_countryCode']);
														$country_details = $this->Country->getCountryAgainstName($user['country']);
														$country_details_short = $this->Country->getCountryAgainstShortName($short_country_name);
														$country_id = count($country_details) ? $country_details['Country']['id'] : ($country_details_short['Country']['id'] ?? 0);
														$state_details = $this->State->getStateAgainstName($state_name, $country_id);
														$state_short_details = $this->State->getStateAgainstShortName($state_short_name, $country_id);
														$state_id = count($state_details) ? $state_details['State']['id'] : ($state_short_details['State']['id'] ?? 0);
														$city_details = $this->City->getCityAgainstName($user['city'], $state_id, $country_id);
														$city_id = $city_details['City']['id'] ?? 0;
													}
													$user['country_id'] = $country_id;
													$user['state_id'] = $state_id;
													$user['city_id'] = $city_id;
													$userDetails = $this->User->getUserDetailsFromID($user_id);
													if (count($userDetails) > 0) {
														$this->User->id = $user_id;
														$this->User->save($user);
														if (count($this->PushNotification->getDetails($user_id)) < 1) {
															$this->PushNotification->save([
																'likes' => 1, 'comments' => 1, 'new_followers' => 1,
																'mentions' => 1, 'direct_messages' => 1,
																'video_updates' => 1, 'id' => $user_id
															]);
														}
														if (count($this->PrivacySetting->getDetails($user_id)) < 1) {
															$this->PrivacySetting->save([
																'videos_download' => 1, 'videos_repost' => 1,
																'direct_message' => 'everyone', 'duet' => 'everyone',
																'liked_videos' => 'me', 'video_comment' => 'everyone',
																'id' => $user_id
															]);
														}
														$userDetails = $this->User->getUserDetailsFromID($user_id);
														return $userDetails;
													}
													return true;
												}

												public function repostVideo(): void
												{
													$this->loadModel('Notification');
													$this->loadModel('RepostVideo');
													$this->loadModel('Video');
													if ($this->getRequest()->is('post')) {
														$data = $this->getRequest()->getData();
														$video_id = $data['video_id'];
														$user_details = $this->user_details;
														if (empty($user_details)) {
															Message::UNAUTHORIZED();
															die();
														}
														$user_id = $user_details['User']['id'];
														$video_details = $this->Video->getOnlyVideoDetails($video_id);
														if (empty($video_details)) {
															Message::DONOTEXIST('Video');
															die();
														}
														$user_cache = $this->getUserDetailsFromCache($video_details['Video']['user_id']);
														$post_data = ['user_id' => $user_id, 'video_id' => $video_id, 'created' => date('Y-m-d H:i:s')];
														$ifExist = $this->RepostVideo->ifExist($post_data);
														if (empty($ifExist)) {
															$this->RepostVideo->save($post_data);
															$details = $this->RepostVideo->getDetails($this->RepostVideo->id);
															if ($user_id !== $details['Video']['user_id']) {
																$msg = $user_details['User']['username'] . ' has reposted your video';
																if (strlen($user_cache['User']['device_token']) > 8) {
																	$notification = [
																		'to' => $user_cache['User']['device_token'],
																		'notification' => [
																			'title' => 'video reposted',
																			'body' => $msg,
																			'badge' => '1',
																			'sound' => 'default',
																			'type' => 'video_reposted',
																			'video_id' => $video_id,
																			'name' => $user_details['User']['first_name'] . ' ' . $user_details['User']['last_name'],
																			'image' => $user_details['User']['profile_pic']
																		],
																		'data' => [
																			'title' => 'video reposted',
																			'body' => $msg,
																			'badge' => '1',
																			'sound' => 'default',
																			'type' => 'video_reposted',
																			'video_id' => $video_id,
																			'name' => $user_details['User']['first_name'] . ' ' . $user_details['User']['last_name'],
																			'image' => $user_details['User']['profile_pic']
																		]
																	];
																	$this->firebase->sendPushNotification($notification);
																	$this->Notification->save([
																		'sender_id' => $user_id,
																		'receiver_id' => $user_cache['User']['id'],
																		'type' => 'video_reposted',
																		'video_id' => $video_id,
																		'string' => $msg,
																		'created' => date('Y-m-d H:i:s')
																	]);
																	Cache::delete('notifications_' . $user_cache['User']['id']);
																}
															}
															echo json_encode(['code' => 200, 'msg' => 'reposted']);
															die();
														} else {
															$this->RepostVideo->id = $ifExist['RepostVideo']['id'];
															$this->RepostVideo->delete();
															echo json_encode(['code' => 201, 'msg' => 'Unpost']);
															die();
														}
													}
												}

												public function placeOrder(): void
												{
													$this->loadModel('Order');
													$this->loadModel('User');
													$this->loadModel('OrderProduct');
													$this->loadModel('Product');
													$this->loadModel('DeliveryAddress');
													$this->loadModel('OrderTransaction');
													$this->loadModel('CouponUsed');
													if ($this->getRequest()->is('post')) {
														$data = $this->getRequest()->getData();
														$user_details = $this->user_details;
														if (empty($user_details)) {
															Message::UNAUTHORIZED();
															die();
														}
														$user_id = $user_details['User']['id'];
														$order = [
															'user_id' => $user_id,
															'store_user_id' => $data['store_user_id'],
															'status' => 0,
															'created' => date('Y-m-d H:i:s'),
															'discount' => $data['discount'],
															'payment_method_id' => 0,
															'cod' => $data['cod'],
															'total' => $data['total'],
															'version' => $data['version'] ?? null,
															'delivery_address_id' => $data['delivery_address_id'],
															'device' => $data['device'] ?? null,
															'delivery' => $data['delivery'],
															'instruction' => $data['instruction'],
															'delivery_fee' => $data['delivery_fee']
														];
														if (empty($this->User->getUserDetailsFromID($user_id))) {
															echo Message::ERROR();
															die();
														}
														$this->Order->save($order);
														$order_id = $this->Order->getInsertID();
														$products = $data['products'];
														foreach ($products as $product) {
															$pd = $this->Product->getDetails($product['product_id']);
															if ($pd) {
																$op = [
																	'product_id' => $product['product_id'],
																	'order_id' => $order_id,
																	'product_title' => $pd['Product']['title'],
																	'product_price' => $pd['Product']['price'],
																	'product_quantity' => $product['product_quantity']
																];
																if (!empty($pd['ProductImage'])) {
																	$op['product_image'] = $pd['ProductImage'][0]['image'];
																}
																$this->OrderProduct->save($op);
															}
														}
														if (!empty($data['coupon_id'])) {
															$this->CouponUsed->save([
																'coupon_id' => $data['coupon_id'],
																'order_id' => $order_id,
																'user_id' => $user_id,
																'created' => date('Y-m-d H:i:s')
															]);
														}
														$order_detail = $this->Order->getDetails($order_id);
														echo json_encode(['code' => 200, 'msg' => $order_detail]);
														die();
													}
												}

												public function addUserImage(): void
												{
													$this->loadModel('User');
													if ($this->getRequest()->is('post')) {
														$ffmpeg = new Ffmpeg();
														$data = $this->getRequest()->getData();
														$user_id = $data['user_id'];
														$extension = $data['extension'];
														$userDetails = $this->User->getUserDetailsFromID($user_id);
														if (empty($userDetails)) {
															echo json_encode(['code' => 201, 'msg' => 'user do not exist']);
															return;
														}
														Cache::delete('user_details_' . $user_id);
														$file_url = $extension === 'gif' ? $userDetails['User']['profile_gif'] : $userDetails['User']['profile_pic'];
														if (strlen($file_url) > 5 && strpos($file_url, 'http') === false && file_exists($file_url)) {
															@unlink($file_url);
														}
														$folder_url = UPLOADS_FOLDER_URI;
														if (MEDIA_STORAGE === 's3') {
															if ($extension === 'mp4') {
																$video_url = Utility::uploadAMultipartFileIntoFolder('file', 'mp4', $folder_url);
																$gif_url = $ffmpeg->videoToGif($video_url, $user_id);
																$result = $this->aws->fileUploadToS3($gif_url, 'gif', $folder_url . '/profile');
																$video_result = $this->aws->fileUploadToS3($video_url, 'mp4', $folder_url . '/profile');
																$file_url = strpos($result['msg'], 'http') !== false ? Utility::getCloudFrontUrl($result['msg'], '/profile') : CLOUDFRONT_URL . '/' . $result['msg'];
															} else {
																$result = $this->aws->fileUploadToS3Multipart('file', $extension);
																$file_url = strpos($result['msg'], 'http') !== false ? Utility::getCloudFrontUrl($result['msg'], '/profile') : CLOUDFRONT_URL . '/' . $result['msg'];
															}
														} else {
															$file_url = Utility::uploadAMultipartFileIntoFolder('file', $extension, $folder_url);
															if ($extension === 'mp4') {
																$file_url = $ffmpeg->videoToGif($file_url, $user_id);
															}
														}
														if ($extension === 'mp4') {
															$user['profile_gif'] = $file_url;
														} else {
															$user['profile_pic'] = $file_url;
														}
														if (strlen(DEEPENGIN_KEY) > 5) {
															$nudity = Utility::verifyPhoto($file_url);
															if ($nudity['code'] === 200 && !empty($nudity['msg']['ModerationResult'])) {
																$n = strtolower($nudity['msg']['ModerationResult']['Name']);
																$p = strtolower($nudity['msg']['ModerationResult']['ParentName']);
																if (Utility::ifWordFound($n) || Utility::ifWordFound($p)) {
																	echo json_encode(['code' => 201, 'msg' => 'Nudity found. Your picture cannot be updated']);
																	return;
																}
															}
														}
														$this->User->id = $user_id;
														if (!$this->User->save($user)) {
															echo Message::DATASAVEERROR();
															return;
														}
														$updated = $this->User->getUserDetailsFromID($user_id);
														echo json_encode(['code' => 200, 'msg' => $updated]);
													}
												}

												public function ttest(): void
												{
													echo 'ss';
												}

												public function addUserImageNew(): void
												{
													$this->loadModel('User');
													if ($this->getRequest()->is('post')) {
														$data = $this->getRequest()->getData();
														$user_id = $data['user_id'];
														$extension = $data['extension'];
														$userDetails = $this->User->getUserDetailsFromID($user_id);
														if (empty($userDetails)) {
															Message::EMPTYDATA();
															die();
														}
														$old_big = $userDetails['User']['profile_pic'];
														$old_small = $userDetails['User']['profile_pic_small'];
														if (strlen($old_big) > 5 && strpos($old_big, 'http') === false) {
															@unlink($old_big);
															@unlink($old_small);
														}
														$folder = UPLOADS_FOLDER_URI;
														$big = Utility::uploadFileintoFolderDir($data['profile_pic'], $folder, $extension);
														$small = Utility::uploadFileintoFolderDir($data['profile_pic_small'], $folder, $extension);
														if (MEDIA_STORAGE === 's3' && method_exists($this->aws, 'profileImageToS3')) {
															$big_res = $this->aws->profileImageToS3($big, $extension === 'jpg' ? 'jpeg' : $extension);
															$small_res = $this->aws->profileImageToS3($small, $extension === 'jpg' ? 'jpeg' : $extension);
															$user['profile_pic'] = Utility::getCloudFrontUrl($big_res['msg'], '/profile');
															$user['profile_pic_small'] = Utility::getCloudFrontUrl($small_res['msg'], '/profile');
															@unlink($big);
															@unlink($small);
														} else {
															$user['profile_pic'] = BASE_URL . $big;
															$user['profile_pic_small'] = BASE_URL . $small;
														}
														if (strlen(DEEPENGIN_KEY) > 5) {
															$check = Utility::verifyPhoto($user['profile_pic']);
															if ($check['code'] === 200 && !empty($check['msg']['ModerationResult'])) {
																$n = strtolower($check['msg']['ModerationResult']['Name']);
																$p = strtolower($check['msg']['ModerationResult']['ParentName']);
																if (Utility::ifWordFound($n) || Utility::ifWordFound($p)) {
																	echo json_encode(['code' => 201, 'msg' => 'Nudity found. Your picture cannot be updated']);
																	return;
																}
															}
														}
														$this->User->id = $user_id;
														$this->User->save($user);
														$updated = $this->User->getUserDetailsFromID($user_id);
														echo json_encode(['code' => 200, 'msg' => $updated]);
													}
												}

												public function shareVideo(): void
												{
													$this->loadModel('Video');
													if ($this->getRequest()->is('post')) {
														$data = $this->getRequest()->getData();
														$video_id = $data['video_id'];
														$video = $this->Video->getOnlyVideoDetails($video_id);
														if (!empty($video)) {
															$this->Video->id = $video_id;
															$this->Video->saveField('share', $video['Video']['share'] + 1);
															$updated = $this->Video->getOnlyVideoDetails($video_id);
															echo json_encode(['code' => 200, 'msg' => $updated]);
														} else {
															echo json_encode(['code' => 201, 'msg' => 'Invalid id: Do not exist']);
														}
													}
												}

												public function addUserProfileVideo(): void
												{
													$this->loadModel('User');
													if ($this->getRequest()->is('post')) {
														$ffmpeg = new Ffmpeg();
														$data = $this->getRequest()->getData();
														$user_id = $data['user_id'];
														$userDetails = $this->User->getUserDetailsFromID($user_id);
														if (empty($userDetails)) {
															Message::EMPTYDATA();
															die();
														}
														$old = $userDetails['User']['profile_gif'];
														if (strlen($old) > 5) {
															if (strpos($old, 'http') === false) {
																@unlink($old);
															} else {
																$this->aws->deleteObjectS3($old);
															}
														}
														$file = Utility::uploadFileintoFolderDir('', '', 'mp4');
														$gif = $ffmpeg->videoToGif($file['msg'], $user_id);
														@unlink($file['msg']);
														if (MEDIA_STORAGE === 's3' && method_exists($this->aws, 'profileImageToS3')) {
															$res = $this->aws->profileImageToS3($gif, 'gif');
															if ($res['code'] === 200) {
																@unlink($gif);
																$gif = Utility::getCloudFrontUrl($res['msg'], '/profile');
															}
														}
														$this->User->id = $user_id;
														$this->User->save(['profile_gif' => $gif]);
														$updated = $this->User->getUserDetailsFromID($user_id);
														echo json_encode(['code' => 200, 'msg' => $updated]);
													}
												}
    
															public function userVerificationRequest(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('VerificationRequests');
																	$this->loadModel('Users');
																	$created = date('Y-m-d H:i:s');
																	$userId = $data['user_id'] ?? null;
																	$existing = $this->VerificationRequests->find()->where(['user_id' => $userId])->first();
																	if (!empty($data['auth_token'])) {
																		$auth = $this->Users->find()->where(['auth_token' => $data['auth_token']])->first();
																		if (!$auth) {
																			return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'']));
																		}
																		$userId = $auth->id;
																	}
																	if (empty($data['attachment'])) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'please send the correct image']));
																	}
																	$filePath = Utility::uploadFileintoFolder($userId, $data['attachment'], UPLOADS_FOLDER_URI);
																	$ext = pathinfo($filePath, PATHINFO_EXTENSION);
																	$up = $this->aws->fileUploadToS3($filePath, $ext);
																	$fileUrl = $up['code']===200 ? Utility::getCloudFrontUrl($up['msg'], '/images') : '';
																	$fields = [
																		'user_id' => $userId,
																		'name'    => $data['name'],
																		'ssn'     => $data['ssn'],
																		'attachment'  => $fileUrl,
																		'update_time' => $created
																	];
																	if ($existing) {
																		if (!empty($existing->attachment)) @unlink($existing->attachment);
																		$entity = $this->VerificationRequests->patchEntity($existing, $fields);
																	} else {
																		$fields['created'] = $created;
																		$entity = $this->VerificationRequests->newEntity($fields);
																	}
																	$this->VerificationRequests->save($entity);
																	unlink($filePath);
																	$saved = $this->VerificationRequests->get($entity->id);
																	$user = $this->Users->get($userId);
																	$user->verification_applied = $this->VerificationRequests->exists(['user_id'=>$userId]) ? 1 : 0;
																	$output = ['code'=>200,'msg'=>$saved];
																	return $this->response->withType('application/json')->withStringBody(json_encode($output));
																}

																public function search(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Videos');
																	$this->loadModel('SoundFavourites');
																	$this->loadModel('Users');
																	$this->loadModel('Sounds');
																	$this->loadModel('Hashtags');
																	$this->loadModel('HashtagVideos');
																	$this->loadModel('HashtagFavourites');
																	$this->loadModel('Followers');
																	$userId = $this->Authentication->getIdentity()->id ?? 0;
																	$keyword = str_replace(' ', '', $data['keyword'] ?? '');
																	$type = $data['type'] ?? '';
																	$start = $data['starting_point'] ?? 0;
																	$results = [];
																	if (in_array($type, ['top','user','store'])) {
																		$storeFlag = $type==='store' ? 1 : 0;
																		$users = $this->Users->getSearchResults($keyword, $start, $userId, $storeFlag);
																		foreach ($users as $u) $this->populateUserStats($u, $userId, $u->id);
																		$results = $type==='top' ? ['videos'=>[], 'users'=>$users] : $users;
																	} elseif ($type==='video') {
																		$videos = $this->Videos->getSearchResults($keyword, $start, $userId);
																		if (empty($videos)) return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'']));
																		foreach ($videos as $v) {
																			$this->customizeVideoResponse($v, $userId);
																			unset($v->video_product);
																		}
																		$results = $videos;
																	} elseif (in_array($type, ['sound','sound_favourite'])) {
																		if ($type==='sound') {
																			$sounds = $this->Sounds->getSearchResults($keyword, $start);
																		} else {
																			$sounds = $this->SoundFavourites->searchFavouriteSound($keyword, $start, $userId);
																		}
																		foreach ($sounds as $s) {
																			$s->favourite = $userId ? (int)$this->SoundFavourites->exists(['user_id'=>$userId,'sound_id'=>$s->id]) : 0;
																		}
																		$results = $sounds;
																	} elseif ($type==='hashtag') {
																		$tags = $this->Hashtags->getSearchResults($keyword, $start);
																		foreach ($tags as $t) {
																			$t->videos_count = $this->HashtagVideos->countHashtagVideos($t->id);
																			$views = $this->HashtagVideos->countHashtagViews($t->id);
																			$t->views = $views[0]->total_sum ?? 0;
																			$t->favourite = (int)$this->HashtagFavourites->exists(['hashtag_id'=>$t->id,'user_id'=>$userId]);
																		}
																		$results = $tags;
																	} elseif (in_array($type, ['following','follower'])) {
																		$list = $type==='following'
																			? $this->Followers->searchFollowing($keyword, $start, $userId)
																			: $this->Followers->searchFollower($keyword, $start, $userId);
																		foreach ($list as $item) {
																			$entity = $type==='following' ? $item->following_list : $item->follower_list;
																			$item->user = $entity;
																			$this->populateUserStats($item, $userId, $entity->id);
																		}
																		$results = $list;
																	}
																	if (empty($results)) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'']));
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$results]));
																}

																public function searchFollowingOrFollowUsers(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Followers');
																	$this->loadModel('Videos');
																	$keyword = $data['keyword'] ?? '';
																	$start = $data['starting_point'] ?? 0;
																	$userId = $data['user_id'] ?? 0;
																	$combined = [];
																	$followers = $this->Followers->searchFollower($keyword, $start, $userId);
																	foreach ($followers as $f) {
																		$pid = $f->follower_list->id;
																		$f->follower_list->follower_count = $this->Followers->countFollowers($pid);
																		$f->follower_list->video_count = $this->Videos->getUserVideosCount($pid);
																		$f->follower_list->button = $this->Followers->getFollowButton($userId, $pid);
																		$combined[] = $f->follower_list;
																	}
																	$following = $this->Followers->searchFollowing($keyword, $start, $userId);
																	foreach ($following as $f) {
																		$pid = $f->following_list->id;
																		$f->following_list->following_count = $this->Followers->countFollowing($pid);
																		$f->following_list->video_count = $this->Videos->getUserVideosCount($pid);
																		$f->following_list->button = $this->Followers->getFollowButton($userId, $pid);
																		$combined[] = $f->following_list;
																	}
																	if (empty($combined)) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'']));
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$combined]));
																}

																public function deleteVideoComment(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('VideoComments');
																	$this->loadModel('Videos');
																	$comment = $this->VideoComments->get($data['id'], ['contain'=>['Videos']]);
																	if (!$comment) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'Invalid id: Do not exist']));
																	}
																	$this->VideoComments->delete($comment);
																	$videoId = $comment->video->id;
																	$video = $this->Videos->get($videoId);
																	$video->pin_comment_id = 0;
																	$this->Videos->save($video);
																	Cache::delete("videocomments_{$videoId}");
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>'deleted']));
																}

																public function showAddSettings(): ResponseInterface
																{
																	$settings = [
																		'followers' => FOLLOWER_PER_COIN,
																		'website_visits' => WEBSITE_VISITS_PER_COIN,
																		'video_views' => VIDEO_VIEWS_PER_COIN
																	];
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$settings]));
																}

																public function forgotPassword(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Users');
																	$user = $this->Users->find()->where(['email'=>$data['email']])->first();
																	if (!$user) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>"Email doesn't exist"]));
																	}
																	$code = Utility::randomNumber(4);
																	$emailData = [
																		'to'=>$user->email,
																		'name'=>$user->first_name.' '.$user->last_name,
																		'subject'=>"reset your password",
																		'message'=>"<p>You requested a password reset. Code: <b>{$code}</b></p>"
																	];
																	$resp = Utility::sendMail($emailData);
																	if ($resp['code']===200) {
																		$user->token = $code;
																		$this->Users->save($user);
																		$output = ['code'=>200,'msg'=>"An email has been sent to {$user->email}. You should receive it shortly."];
																	} else {
																		$output = ['code'=>201,'msg'=>$resp['msg']];
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode($output));
																}

																public function verifyforgotPasswordCode(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Users');
																	$user = $this->Users->find()->where(['email'=>$data['email'],'token'=>$data['code']])->first();
																	if (!$user) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'invalid code']));
																	}
																	$user->token = null;
																	$this->Users->save($user);
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$user]));
																}

																public function verifyAccountDetails(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Users');
																	if ($data['type']==='phone') {
																		$exists = $this->Users->exists(['phone'=>$data['social_id']]);
																	} else {
																		$exists = $this->Users->exists(['social_id'=>$data['social_id']]);
																	}
																	if ($exists) {
																		$user = $this->Users->find()->where($data['type']==='phone' ? ['phone'=>$data['social_id']] : ['social_id'=>$data['social_id']])->first();
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$user]));
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'']));
																}

																public function changePassword(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Users');
																	$user = $this->Users->get($data['user_id']);
																	if (!$this->Users->checkPassword($user, $data['old_password'])) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'incorrect password']));
																	}
																	$user->password = $data['new_password'];
																	if ($this->Users->save($user)) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>'Password changed successfully']));
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'Unable to save new password']));
																}

																public function updateProfile(): ResponseInterface
																{
																	$this->loadModel('Users');
																	$users = $this->Users->find()->where(['has_profile_pic'=>1])->all();
																	foreach ($users as $u) {
																		$changed = false;
																		if (strpos($u->profile_pic, 'http')===false && !empty($u->profile_pic)) {
																			$big = $this->aws->profileImageToS3($u->profile_pic,'png');
																			if ($big['code']===200) {
																				$u->profile_pic = Utility::getCloudFrontUrl($big['msg'],'/profile');
																				@unlink($u->profile_pic);
																				$changed = true;
																			}
																		}
																		if (strpos($u->profile_pic_small, 'http')===false && !empty($u->profile_pic_small)) {
																			$small = $this->aws->profileImageToS3($u->profile_pic_small,'png');
																			if ($small['code']===200) {
																				$u->profile_pic_small = Utility::getCloudFrontUrl($small['msg'],'/profile');
																				@unlink($u->profile_pic_small);
																				$changed = true;
																			}
																		}
																		if ($changed) {
																			$this->Users->save($u);
																		}
																	}
																	return $this->response;
																}

																public function changeEmailAddress(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('Users');
																	$user = $this->Users->get($data['user_id']);
																	if ($user->email === $data['email']) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$user]));
																	}
																	if ($this->Users->exists(['email'=>$data['email']])) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'This email has already been registered']));
																	}
																	$code = Utility::randomNumber(4);
																	$emailData = [
																		'to'=>$data['email'],
																		'name'=>$user->first_name.' '.$user->last_name,
																		'subject'=>"change your email address",
																		'message'=>"<p>Please enter this verification code to reset your email. Code: <b>{$code}</b></p>"
																	];
																	$resp = Utility::sendMail($emailData);
																	if ($resp['code']===200) {
																		$user->token = $code;
																		$this->Users->save($user);
																		$output = ['code'=>200,'msg'=>"An email has been sent to {$data['email']}."];
																	} else {
																		$output = ['code'=>201,'msg'=>$resp['msg']];
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode($output));
																}

																public function verifyRegisterEmailCode(): ResponseInterface
																{
																	$this->request->allowMethod(['post']);
																	$data = $this->getRequest()->getData();
																	$this->loadModel('EmailVerifications');
																	$this->loadModel('Users');
																	if ($this->Users->exists(['email'=>$data['email']])) {
																		$existing = $this->Users->find()->where(['email'=>$data['email']])->first();
																		if ($existing->active>1) {
																			return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'You have been blocked by the admin. Contact support']));
																		}
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'The account already exists with this email']));
																	}
																	if (empty($data['code'])) {
																		$code = (APP_STATUS==='live') ? Utility::randomNumber(4) : '1234';
																		$entity = $this->EmailVerifications->newEntity(['email'=>$data['email'],'code'=>$code]);
																		$this->EmailVerifications->save($entity);
																		if (APP_STATUS==='live') Utility::sendMail(['to'=>$data['email'],'subject'=>'verify your email address','message'=>"<p>Confirmation code: <b>{$code}</b></p>"]);
																		$saved = $this->EmailVerifications->get($entity->id);
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$saved]));
																	}
																	$valid = $this->EmailVerifications->find()->where(['email'=>$data['email'],'code'=>$data['code']])->first();
																	if ($valid) {
																		return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>200,'msg'=>$valid]));
																	}
																	return $this->response->withType('application/json')->withStringBody(json_encode(['code'=>201,'msg'=>'invalid code']));
																}
    
																	public function verifyChangeEmailCode(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$code = $data['code'] ?? null;
																		$email = $data['new_email'] ?? null;
																		$userId = $data['user_id'] ?? null;
																		$userDetails = $this->User->getUserDetailsFromID($userId);
																		if ($userDetails) {
																			$dbEmail = $userDetails['User']['email'];
																			$codeVerify = $this->User->verifyToken($code, $dbEmail);
																			if ($codeVerify && $code > 0) {
																				$entity = $this->User->get($userId);
																				$entity->email = $email;
																				$entity->token = 0;
																				$entity->active = 1;
																				$this->User->save($entity);
																				$updatedUser = $this->User->getUserDetailsFromEmail($email);
																				$result = ['code' => 200, 'msg' => $updatedUser];
																				return $this->response->withType('application/json')->withStringBody(json_encode($result));
																			}
																		}
																		$result = ['code' => 201, 'msg' => 'invalid code'];
																		return $this->response->withType('application/json')->withStringBody(json_encode($result));
																	}

																	public function changePhoneNo(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$userId = $data['user_id'] ?? null;
																		$phone = $data['phone'] ?? null;
																		$phoneExist = $this->User->editIsphoneNoAlreadyExist($phone, $userId);
																		$userDetails = $this->User->getUserDetailsFromID($userId);
																		if ($userDetails) {
																			$dbPhone = $userDetails['User']['phone'];
																			if ($dbPhone === $phone) {
																				$result = ['code' => 200, 'msg' => $userDetails];
																				return $this->response->withType('application/json')->withStringBody(json_encode($result));
																			}
																			if ($phoneExist > 0) {
																				$result = ['code' => 201, 'msg' => 'This phone has already been registered'];
																				return $this->response->withType('application/json')->withStringBody(json_encode($result));
																			}
																			return $this->verifyPhoneNo($phone, $userId, 0);
																		}
																		return $this->response->withStatus(404);
																	}

																	public function verifyPhoneNo(?string $phoneNo = null, ?int $userId = null, ?int $verify = null): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('PhoneNoVerification');
																		$this->loadModel('User');
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		if ($phoneNo === null) {
																			$phoneNo = $data['phone'] ?? null;
																			$verify = $data['verify'] ?? null;
																			if (isset($data['user_id'])) {
																				$userId = $data['user_id'];
																				$phoneExist = $this->User->editIsphoneNoAlreadyExist($phoneNo, $userId);
																				if ($phoneExist > 0) {
																					$result = ['code' => 201, 'msg' => 'This phone has already been registered'];
																					return $this->response->withType('application/json')->withStringBody(json_encode($result));
																				}
																			}
																		}
																		$code = Utility::randomNumber(4);
																		if (APP_STATUS === 'demo') {
																			$code = 1234;
																		}
																		$created = date('Y-m-d H:i:s', time() - 14400);
																		$phoneVerify = ['phone_no' => $phoneNo, 'code' => $code, 'created' => $created];
																		if ($verify === 0) {
																			if (APP_STATUS === 'demo') {
																				$response = ['sid' => ''];
																			} else {
																				$response = Utility::sendSmsVerificationCurl($phoneNo, VERIFICATION_PHONENO_MESSAGE . ' ' . $code);
																			}
																			if (isset($response['code'])) {
																				$output = ['code' => 201, 'msg' => $response['message']];
																			} elseif (isset($response['sid'])) {
																				$this->PhoneNoVerification->save($this->PhoneNoVerification->newEntity($phoneVerify));
																				$output = ['code' => 200, 'msg' => "code has been generated and sent to user's phone number"];
																			}
																		} else {
																			$codeUser = $data['code'] ?? null;
																			if ($this->PhoneNoVerification->verifyCode($phoneNo, $codeUser) > 0) {
																				if ($userId) {
																					$entity = $this->User->get($userId);
																					$entity->phone = $phoneNo;
																					$this->User->save($entity);
																				}
																				$output = ['code' => 200, 'msg' => 'successfully code matched'];
																			} else {
																				$output = ['code' => 201, 'msg' => 'invalid code'];
																			}
																		}
																		return $this->response->withType('application/json')->withStringBody(json_encode($output));
																	}

																	public function blockUser(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('BlockUser');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$userId = $data['user_id'] ?? null;
																		$blockUserId = $data['block_user_id'] ?? null;
																		$record = ['created' => date('Y-m-d H:i:s'), 'user_id' => $userId, 'block_user_id' => $blockUserId];
																		$existing = $this->BlockUser->ifAlreadyBlocked($record);
																		if (!$existing) {
																			$entity = $this->BlockUser->newEntity($record);
																			$this->BlockUser->save($entity);
																			$details = $this->BlockUser->getDetails($entity->id);
																			$result = ['code' => 200, 'msg' => $details];
																		} else {
																			$entity = $this->BlockUser->get($existing['BlockUser']['id']);
																			$this->BlockUser->delete($entity);
																			$result = ['code' => 201, 'msg' => 'deleted'];
																		}
																		return $this->response->withType('application/json')->withStringBody(json_encode($result));
																	}

																	public function showBlockedUsers(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('BlockUser');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$userId = $data['user_id'] ?? null;
																		$details = $this->BlockUser->getBlockUsers($userId);
																		if ($details) {
																			$result = ['code' => 200, 'msg' => $details];
																			return $this->response->withType('application/json')->withStringBody(json_encode($result));
																		}
																		return $this->response->withType('application/json')->withStringBody(Message::EMPTYDATA());
																	}

																	public function showPayout(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('Payout');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$userId = $data['user_id'] ?? null;
																		$details = $this->Payout->ifExist($userId);
																		if ($details) {
																			$result = ['code' => 200, 'msg' => $details];
																			return $this->response->withType('application/json')->withStringBody(json_encode($result));
																		}
																		return $this->response->withType('application/json')->withStringBody(Message::EMPTYDATA());
																	}

																	public function changePasswordForgot(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		if (!$this->request->is('post')) {
																			return $this->response->withType('application/json')->withStringBody(Message::INCORRECTPASSWORD());
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$email = $data['email'] ?? null;
																		$newPassword = $data['password'] ?? null;
																		$userDetails = $this->User->getUserDetailsAgainstEmail($email);
																		$userId = $userDetails['User']['id'] ?? null;
																		$entity = $this->User->get($userId);
																		$entity->password = $newPassword;
																		if ($this->User->save($entity)) {
																			$updatedUser = $this->User->getUserDetailsFromID($userId);
																			$result = ['code' => 200, 'msg' => $updatedUser];
																			return $this->response->withType('application/json')->withStringBody(json_encode($result));
																		}
																		return $this->response->withType('application/json')->withStringBody(Message::DATASAVEERROR());
																	}

																	public function sendLiveStreamPushNotfication(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		$this->loadModel('PushNotification');
																		$this->loadModel('Follower');
																		$this->loadModel('Notification');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$userId = $data['user_id'] ?? null;
																		$liveStreamId = $data['live_streaming_id'] ?? null;
																		$userDetails = $this->User->getUserDetailsFromID($userId);
																		$followers = $this->Follower->getUserFollowersWithoutLimit($userId);
																		foreach ($followers ?: [] as $follower) {
																			$deviceToken = $follower['FollowerList']['device_token'];
																			$receiverId = $follower['FollowerList']['id'];
																			$msg = $userDetails['User']['username'] . ' is live now';
																			if (strlen($deviceToken) > 8) {
																				$notification = [
																					'to' => $deviceToken,
																					'notification' => [
																						'title' => $userDetails['User']['username'],
																						'body' => $msg,
																						'badge' => '1',
																						'sound' => 'default',
																						'icon' => '',
																						'type' => 'live',
																						'user_id' => $userId,
																						'name' => $userDetails['User']['first_name'] . ' ' . $userDetails['User']['last_name'],
																						'image' => $userDetails['User']['profile_pic'],
																						'receiver_id' => $receiverId
																					],
																					'data' => [
																						'title' => $userDetails['User']['username'],
																						'body' => $msg,
																						'badge' => '1',
																						'sound' => 'default',
																						'icon' => '',
																						'type' => 'live',
																						'user_id' => $userId,
																						'name' => $userDetails['User']['first_name'] . ' ' . $userDetails['User']['last_name'],
																						'image' => $userDetails['User']['profile_pic'],
																						'receiver_id' => $receiverId
																					]
																				];
																				$this->firebase->sendPushNotification($notification);
																				$notificationData = [
																					'sender_id' => $userId,
																					'receiver_id' => $receiverId,
																					'type' => 'live',
																					'live_streaming_id' => $liveStreamId,
																					'string' => $msg,
																					'created' => date('Y-m-d H:i:s')
																				];
																				$entity = $this->Notification->newEntity($notificationData);
																				$this->Notification->save($entity);
																				Cache::delete('notifications_' . $receiverId);
																			}
																		}
																		$result = ['code' => 200, 'msg' => 'success'];
																		return $this->response->withType('application/json')->withStringBody(json_encode($result));
																	}

																	public function deleteUserAccount(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		$this->loadModel('Follower');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$userDetails = $this->user_details ?? null;
																		if (!$userDetails) {
																			return $this->response->withType('application/json')->withStringBody(Message::UNAUTHORIZED())->withStatus(401);
																		}
																		$userId = $userDetails['User']['id'];
																		$entity = $this->User->get($userId);
																		$this->User->delete($entity);
																		$this->Follower->deleteFollowerAgainstUserID($userId);
																		$this->firebase->deleteUser($userDetails['User']['auth_token']);
																		Cache::delete('user_details_' . $userId);
																		$result = ['code' => 200, 'msg' => 'deleted'];
																		return $this->response->withType('application/json')->withStringBody(json_encode($result));
																	}

																	public function deleteFollower(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		$this->loadModel('Follower');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$followerId = $data['follower_id'] ?? null;
																		$userId = $data['user_id'] ?? null;
																		$details = $this->Follower->ifFollowing($followerId, $userId);
																		if ($details) {
																			$entity = $this->Follower->get($details['Follower']['id']);
																			$this->Follower->delete($entity);
																			$result = ['code' => 200, 'msg' => 'deleted'];
																		} else {
																			$result = ['code' => 201, 'msg' => 'Invalid id: Do not exist'];
																		}
																		return $this->response->withType('application/json')->withStringBody(json_encode($result));
																	}

																	public function sendMessageNotification(): \Psr\Http\Message\ResponseInterface
																	{
																		$this->loadModel('User');
																		$this->loadModel('PushNotification');
																		if (!$this->request->is('post')) {
																			return $this->response->withStatus(405);
																		}
																		$data = json_decode((string)$this->request->getBody(), true, 512, JSON_THROW_ON_ERROR);
																		$senderId = $data['sender_id'] ?? null;
																		$title = $data['title'] ?? null;
																		$message = $data['message'] ?? null;
																		$senderDetails = $this->User->getUserDetailsFromID($senderId);
																		if (isset($data['receivers'])) {
																			foreach ($data['receivers'] as $receiver) {
																				$receiverId = $receiver['receiver_id'];
																				$receiverDetails = $this->User->getUserDetailsFromID($receiverId);
																				$notification = [
																					'to' => $receiverDetails['User']['device_token'],
																					'notification' => [
																						'title' => $title,
																						'body' => $message,
																						'user_id' => $senderDetails['User']['id'],
																						'image' => $senderDetails['User']['profile_pic'],
																						'name' => $senderDetails['User']['username'],
																						'badge' => '1',
																						'sound' => 'default',
																						'icon' => '',
																						'type' => 'message',
																						'receiver_id' => $receiverDetails['User']['id']
																					],
																					'data' => [
																						'title' => $title,
																						'name' => $senderDetails['User']['username'],
																						'body' => $message,
																						'icon' => '',
																						'badge' => '1',
																						'sound' => 'default',
																						'type' => 'message',
																						'user_id' => $senderDetails['User']['id'],
																						'image' => $senderDetails['User']['profile_pic'],
																						'receiver_id' => $receiverDetails['User']['id']
																					]
																				];
																				$existing = $this->PushNotification->getDetails($receiverDetails['User']['id']);
																				if ($existing && $existing['PushNotification']['direct_messages'] > 0) {
																					$this->firebase->sendPushNotification($notification);
																				}
																			}
																		}
																		if (isset($data['receiver_id'])) {
																			$receiverId = $data['receiver_id'];
																			$receiverDetails = $this->User->getUserDetailsFromID($receiverId);
																			$notification = [
																				'to' => $receiverDetails['User']['device_token'],
																				'notification' => [
																					'title' => $title,
																					'body' => $message,
																					'user_id' => $senderDetails['User']['id'],
																					'image' => $senderDetails['User']['profile_pic'],
																					'name' => $senderDetails['User']['username'],
																					'badge' => '1',
																					'sound' => 'default',
																					'icon' => '',
																					'type' => 'message',
																					'receiver_id' => $receiverDetails['User']['id']
																				],
																				'data' => [
																					'title' => $title,
																					'name' => $senderDetails['User']['username'],
																					'body' => $message,
																					'icon' => '',
																					'badge' => '1',
																					'sound' => 'default',
																					'type' => 'message',
																					'user_id' => $senderDetails['User']['id'],
																					'image' => $senderDetails['User']['profile_pic'],
																					'receiver_id' => $receiverDetails['User']['id']
																				]
																			];
																			$existing = $this->PushNotification->getDetails($receiverDetails['User']['id']);
																			if ($existing && $existing['PushNotification']['direct_messages'] > 0) {
																				$this->firebase->sendPushNotification($notification);
																			}
																		}
																		$result = ['code' => 200, 'msg' => 'success'];
																		return $this->response->withType('application/json')->withStringBody(json_encode($result));
																	}
}
?>
