<?php
declare(strict_types=1);

namespace App\Controller;

use App\Controller\AppController;
use Cake\Core\Configure;
use Cake\Datasource\ConnectionManager;
use Cake\Http\Response;
use App\Utility\Aws;

class ConfigController extends AppController
{
	protected Aws $aws;

	public function initialize(): void
	{
		parent::initialize();
		$this->loadComponent('RequestHandler');
		$this->aws = Aws::getInstance();
	}

	public function phpInfo(): Response
	{
		$this->viewBuilder()->disableAutoLayout();
		$info = phpinfo();
		return $this->getResponse()->withStringBody($info);
	}

	public function view(): void
	{
		$this->viewBuilder()->disableAutoLayout();
		$Videos = $this->getTableLocator()->get('Videos');
		$params = $this->getRequest()->getQueryParams();
		$urlParam = array_key_first($params);
		$videoId = (int)filter_var($urlParam, FILTER_SANITIZE_NUMBER_INT);
		$videoDetail = $Videos->getDetails($videoId);
		$configData = [];
		if (defined('MEDIA_STORAGE') && MEDIA_STORAGE === 's3') {
			$configData['s3'] = $this->aws->fileUploadToS3(
				'webroot/uploads/images/test.jpg',
				'png',
				UPLOADS_FOLDER_URI
			);
		}
		$this->set(compact('configData', 'videoDetail'));
	}

	public function config(): void
	{
		$this->viewBuilder()->disableAutoLayout();
		$configData = [];
		if (defined('MEDIA_STORAGE') && MEDIA_STORAGE === 's3') {
			$configData['s3'] = $this->aws->fileUploadToS3(
				'webroot/uploads/images/test.jpg',
				'png',
				UPLOADS_FOLDER_URI
			);
		}
		$this->set('configData', $configData);
	}

	public function setupPrivacyAndPush(): void
	{
		$Users = $this->getTableLocator()->get('Users');
		$PrivacySettings = $this->getTableLocator()->get('PrivacySettings');
		$PushNotifications = $this->getTableLocator()->get('PushNotifications');

		$users = $Users->find()->all();
		$notifications = [];
		$settings = [];

		foreach ($users as $user) {
			$notifications[] = [
				'user_id' => $user->id,
				'likes' => true,
				'comments' => true,
				'new_followers' => true,
				'mentions' => true,
				'direct_messages' => true,
				'video_updates' => true,
			];
			$settings[] = [
				'user_id' => $user->id,
				'videos_download' => true,
				'videos_repost' => true,
				'direct_message' => 'everyone',
				'duet' => 'everyone',
				'liked_videos' => 'me',
				'video_comment' => 'everyone',
			];
		}

		$notificationEntities = $PushNotifications->newEntities($notifications);
		$PushNotifications->saveMany($notificationEntities);

		$settingEntities = $PrivacySettings->newEntities($settings);
		$PrivacySettings->saveMany($settingEntities);
	}

	protected function findKey(array $array, string $keySearch): bool
	{
		foreach ($array as $key => $item) {
			if ($key === $keySearch || (is_array($item) && $this->findKey($item, $keySearch))) {
				return true;
			}
		}
		return false;
	}

	public function showApiSettings(): void
	{
		$ApiSettings = $this->getTableLocator()->get('ApiSettings');
		$details = $ApiSettings->find()->all();
		if ($details->isEmpty()) {
			return;
		}
		$apiConfig = [];
		foreach ($details as $detail) {
			if ($detail->title === 'API_KEY') {
				$apiConfig['API_KEY'] = $detail->value;
			}
			if ($detail->title === 'API_STATUS') {
				$apiConfig['API_STATUS'] = $detail->value;
			}
		}
		\Cake\Core\Configure::write('ApiSetting', $apiConfig);
	}

	public function checkDatabaseConnection(): bool
	{
		try {
			$connection = ConnectionManager::get('default');
			return $connection->connect();
		} catch (\Exception $e) {
			return false;
		}
	}
}
